#!/bin/bash

# 🚀 Start Restaurant Chatbot Server with ngrok
# This script starts both the server and ngrok, then sets up the webhook

echo "========================================================================"
echo "🚀 Restaurant Chatbot Server - Quick Start with ngrok"
echo "========================================================================"
echo ""

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Check if ngrok is installed
if ! command -v ngrok &> /dev/null; then
    echo -e "${RED}❌ ngrok is not installed!${NC}"
    echo ""
    echo "Please install ngrok:"
    echo "  Mac: brew install ngrok"
    echo "  Linux: https://ngrok.com/download"
    exit 1
fi

# Check if .env exists
if [ ! -f ".env" ]; then
    echo -e "${RED}❌ .env file not found!${NC}"
    echo ""
    echo "Please create .env file with your credentials"
    exit 1
fi

# Create logs directory
mkdir -p logs

echo -e "${GREEN}✅ Starting server...${NC}"
echo ""

# Start server in background
python3 app/main.py > logs/server.log 2>&1 &
SERVER_PID=$!

echo "   Server PID: $SERVER_PID"
echo "   Waiting for server to start..."
sleep 3

# Check if server is running
if ! curl -s http://localhost:8000/health > /dev/null; then
    echo -e "${RED}❌ Server failed to start!${NC}"
    echo "Check logs/server.log for details"
    kill $SERVER_PID 2>/dev/null
    exit 1
fi

echo -e "${GREEN}✅ Server is running on http://localhost:8000${NC}"
echo ""

echo -e "${GREEN}✅ Starting ngrok...${NC}"
echo ""

# Start ngrok in background
ngrok http 8000 --log=stdout > logs/ngrok.log 2>&1 &
NGROK_PID=$!

echo "   ngrok PID: $NGROK_PID"
echo "   Waiting for ngrok to start..."
sleep 5

# Get ngrok URL
echo ""
echo -e "${YELLOW}🔍 Getting ngrok URL...${NC}"
echo ""

NGROK_URL=$(python3 get_ngrok_url.py 2>&1 | grep "Public URL:" | awk '{print $4}')

if [ -z "$NGROK_URL" ]; then
    echo -e "${RED}❌ Failed to get ngrok URL!${NC}"
    echo "Check logs/ngrok.log for details"
    kill $SERVER_PID $NGROK_PID 2>/dev/null
    exit 1
fi

echo -e "${GREEN}✅ ngrok is running!${NC}"
echo ""
echo "========================================================================"
echo "📡 URLs:"
echo "========================================================================"
echo "   🌐 Public URL: $NGROK_URL"
echo "   🔗 Webhook: $NGROK_URL/webhook/whatsapp"
echo "   🖥️  Local Server: http://localhost:8000"
echo "   📊 ngrok Dashboard: http://127.0.0.1:4040"
echo "========================================================================"
echo ""

# Ask if user wants to setup webhook automatically
read -p "Do you want to setup webhook automatically? (y/n): " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    echo -e "${YELLOW}🔧 Setting up webhook...${NC}"
    echo ""
    python3 auto_setup_webhook.py
fi

echo ""
echo "========================================================================"
echo "✅ Everything is ready!"
echo "========================================================================"
echo ""
echo "📝 Logs:"
echo "   • Server: tail -f logs/server.log"
echo "   • App: tail -f logs/app.log"
echo "   • ngrok: tail -f logs/ngrok.log"
echo ""
echo "🛑 To stop:"
echo "   • Press Ctrl+C"
echo "   • Or run: kill $SERVER_PID $NGROK_PID"
echo ""
echo "========================================================================"
echo ""

# Save PIDs to file for easy cleanup
echo "$SERVER_PID" > .server.pid
echo "$NGROK_PID" > .ngrok.pid

# Wait for user to stop
echo "Press Ctrl+C to stop..."
echo ""

# Trap Ctrl+C
trap cleanup INT

cleanup() {
    echo ""
    echo ""
    echo "🛑 Stopping services..."
    kill $SERVER_PID $NGROK_PID 2>/dev/null
    rm -f .server.pid .ngrok.pid
    echo "✅ Stopped!"
    exit 0
}

# Keep script running
tail -f logs/app.log

