"""
Verify Last Customer Data
This script queries Supabase to verify the last customer who contacted the bot
"""
import os
import sys
from datetime import datetime

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from supabase import create_client
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Initialize Supabase
supabase = create_client(
    os.getenv('SUPABASE_URL'),
    os.getenv('SUPABASE_KEY')
)

def print_section(title):
    """Print section header"""
    print("\n" + "="*70)
    print(f"  {title}")
    print("="*70)

def verify_last_customer():
    """Verify last customer data from database"""
    
    # From logs, we know the last customer
    CHAT_ID = "201556307784@s.whatsapp.net"
    EXPECTED_CHAT_UUID = "32c71b32-067d-4954-844b-1cf8e0d04984"
    EXPECTED_ORDER_ID = "10019"
    
    print_section("🔍 VERIFYING LAST CUSTOMER DATA")
    print(f"\n📱 Chat ID: {CHAT_ID}")
    print(f"🆔 Expected Chat UUID: {EXPECTED_CHAT_UUID}")
    print(f"📦 Expected Order ID: {EXPECTED_ORDER_ID}")
    
    # ============================================================
    # 1. Get Chat Info
    # ============================================================
    print_section("1️⃣ CHAT INFO")
    
    try:
        chat_response = supabase.table('chats') \
            .select('*') \
            .eq('chat_id', CHAT_ID) \
            .execute()
        
        if chat_response.data:
            chat = chat_response.data[0]
            print(f"\n✅ Chat found!")
            print(f"   - ID (UUID): {chat['id']}")
            print(f"   - Chat ID: {chat['chat_id']}")
            print(f"   - Phone: {chat.get('phone_number', 'N/A')}")
            print(f"   - Customer Name: {chat.get('customer_name', 'N/A')}")
            print(f"   - Created: {chat.get('created_at', 'N/A')}")
            print(f"   - Updated: {chat.get('updated_at', 'N/A')}")
            
            # Verify UUID matches
            if chat['id'] == EXPECTED_CHAT_UUID:
                print(f"\n   ✅ UUID MATCHES! ({EXPECTED_CHAT_UUID})")
            else:
                print(f"\n   ❌ UUID MISMATCH!")
                print(f"      Expected: {EXPECTED_CHAT_UUID}")
                print(f"      Got: {chat['id']}")
            
            CHAT_UUID = chat['id']
        else:
            print(f"\n❌ Chat not found!")
            return
            
    except Exception as e:
        print(f"\n❌ Error getting chat: {e}")
        return
    
    # ============================================================
    # 2. Get Last Session
    # ============================================================
    print_section("2️⃣ LAST SESSION")
    
    try:
        session_response = supabase.table('draft_sessions') \
            .select('*') \
            .eq('chat_id', CHAT_ID) \
            .order('created_at', desc=True) \
            .limit(1) \
            .execute()
        
        if session_response.data:
            session = session_response.data[0]
            print(f"\n✅ Session found!")
            print(f"   - ID: {session['id']}")
            print(f"   - Stage: {session.get('stage', 'N/A')}")
            print(f"   - Phase: {session.get('phase', 'N/A')}")
            print(f"   - Status: {session.get('status', 'N/A')}")
            print(f"   - Language: {session.get('lang', 'N/A')}")
            print(f"   - Items Count: {len(session.get('items', []))}")
            print(f"   - Created: {session.get('created_at', 'N/A')}")
            print(f"   - Updated: {session.get('updated_at', 'N/A')}")
        else:
            print(f"\n⚠️ No session found")
            
    except Exception as e:
        print(f"\n❌ Error getting session: {e}")
    
    # ============================================================
    # 3. Get Last Order (WITHOUT order_items join)
    # ============================================================
    print_section("3️⃣ LAST ORDER (Basic Info)")
    
    try:
        order_response = supabase.table('orders') \
            .select('id, order_id, total_omr, created_at, status') \
            .eq('chat_id', CHAT_UUID) \
            .order('created_at', desc=True) \
            .limit(1) \
            .execute()
        
        if order_response.data:
            order = order_response.data[0]
            print(f"\n✅ Order found!")
            print(f"   - ID (UUID): {order['id']}")
            print(f"   - Order ID: {order.get('order_id', 'N/A')}")
            print(f"   - Total: {order.get('total_omr', 0):.3f} OMR")
            print(f"   - Status: {order.get('status', 'N/A')}")
            print(f"   - Created: {order.get('created_at', 'N/A')}")
            
            # Verify Order ID matches
            if order.get('order_id') == EXPECTED_ORDER_ID:
                print(f"\n   ✅ ORDER ID MATCHES! ({EXPECTED_ORDER_ID})")
            else:
                print(f"\n   ⚠️ ORDER ID MISMATCH!")
                print(f"      Expected: {EXPECTED_ORDER_ID}")
                print(f"      Got: {order.get('order_id')}")
            
            ORDER_UUID = order['id']
            
            # ============================================================
            # 4. Get Order Items
            # ============================================================
            print_section("4️⃣ ORDER ITEMS")
            
            try:
                items_response = supabase.table('order_items') \
                    .select('*') \
                    .eq('order_id', ORDER_UUID) \
                    .execute()
                
                if items_response.data:
                    print(f"\n✅ Found {len(items_response.data)} item(s):")
                    
                    for i, item in enumerate(items_response.data, 1):
                        print(f"\n   Item {i}:")
                        print(f"      - Name: {item.get('item_name', 'N/A')}")
                        print(f"      - Code: {item.get('item_code', 'N/A')}")
                        print(f"      - Quantity: {item.get('qty', 0)}")
                        print(f"      - Unit Price: {item.get('unit_price_omr', 0):.3f} OMR")
                        print(f"      - Subtotal: {item.get('subtotal_omr', 0):.3f} OMR")
                else:
                    print(f"\n⚠️ No items found for this order")
                    
            except Exception as e:
                print(f"\n❌ Error getting order items: {e}")
                import traceback
                print(traceback.format_exc())
            
        else:
            print(f"\n⚠️ No order found for this customer")
            
    except Exception as e:
        print(f"\n❌ Error getting order: {e}")
        import traceback
        print(traceback.format_exc())
    
    # ============================================================
    # 5. Get Conversation History
    # ============================================================
    print_section("5️⃣ CONVERSATION HISTORY (Last 5 messages)")
    
    try:
        history_response = supabase.table('conversation_history') \
            .select('*') \
            .eq('chat_id', CHAT_ID) \
            .order('created_at', desc=True) \
            .limit(5) \
            .execute()
        
        if history_response.data:
            print(f"\n✅ Found {len(history_response.data)} message(s):")
            
            for i, msg in enumerate(history_response.data, 1):
                role = msg.get('role', 'unknown')
                content = msg.get('content', '')
                created = msg.get('created_at', 'N/A')
                
                # Truncate long messages
                if len(content) > 100:
                    content = content[:100] + "..."
                
                print(f"\n   Message {i}:")
                print(f"      - Role: {role}")
                print(f"      - Content: {content}")
                print(f"      - Time: {created}")
        else:
            print(f"\n⚠️ No conversation history found")
            
    except Exception as e:
        print(f"\n❌ Error getting conversation history: {e}")
    
    # ============================================================
    # Summary
    # ============================================================
    print_section("📊 SUMMARY")
    print(f"\n✅ Verification complete!")
    print(f"\n📱 Customer: {CHAT_ID}")
    print(f"🆔 Chat UUID: {CHAT_UUID}")
    print(f"📦 Last Order: {EXPECTED_ORDER_ID}")
    print(f"\n💡 All data retrieved successfully from database!")


if __name__ == "__main__":
    try:
        verify_last_customer()
    except Exception as e:
        print(f"\n❌ Fatal error: {e}")
        import traceback
        print(traceback.format_exc())

