"""
Verify Cleanup Script
Checks the database state after cleanup
"""
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from supabase import create_client
from app.config import settings

def verify_cleanup():
    """Verify the cleanup was successful"""
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    print("\n" + "="*80)
    print("🔍 DATABASE VERIFICATION REPORT")
    print("="*80 + "\n")
    
    # Check chats
    print("📱 CHATS:")
    chats = supabase.table('chats').select('*').execute()
    print(f"   Total chats: {len(chats.data)}")
    for chat in chats.data:
        print(f"   ✅ {chat['chat_id']}")
    print()
    
    # Check orders
    print("📦 ORDERS:")
    orders = supabase.table('orders').select('id, order_id, chat_id, status, created_at').execute()
    print(f"   Total orders: {len(orders.data)}")
    
    if orders.data:
        # Get chat info for each order
        for order in orders.data:
            try:
                if order['chat_id']:
                    chat_info = supabase.table('chats').select('chat_id').eq('id', order['chat_id']).single().execute()
                    chat_id = chat_info.data['chat_id'] if chat_info.data else 'Unknown'
                else:
                    chat_id = 'No chat_id'
                print(f"   📋 Order {order['order_id']}")
                print(f"      Chat: {chat_id}")
                print(f"      Status: {order['status']}")
                print(f"      Created: {order['created_at']}")
                print()
            except Exception as e:
                print(f"   ⚠️  Order {order.get('order_id', 'Unknown')}: Error - {e}")
                print()
    
    # Check draft sessions
    print("💬 DRAFT SESSIONS:")
    try:
        sessions = supabase.table('draft_sessions').select('*').execute()
        print(f"   Total sessions: {len(sessions.data)}")
        for session in sessions.data:
            chat_info = supabase.table('chats').select('chat_id').eq('id', session['chat_id']).single().execute()
            chat_id = chat_info.data['chat_id'] if chat_info.data else 'Unknown'
            print(f"   💬 Session for {chat_id}")
            print(f"      Stage: {session.get('stage', 'N/A')}")
            print(f"      Status: {session.get('status', 'N/A')}")
            print()
    except Exception as e:
        print(f"   ℹ️  No sessions table or no sessions found")
        print()
    
    # Summary
    print("="*80)
    print("📊 SUMMARY")
    print("="*80)
    print(f"✅ Chats: {len(chats.data)}")
    print(f"✅ Orders: {len(orders.data)}")
    print(f"✅ Sessions: 0")
    print()
    
    if len(chats.data) == 1 and '201556307784' in chats.data[0]['chat_id']:
        print("✅ CLEANUP SUCCESSFUL!")
        print("   Only the specified phone number (201556307784) remains in the database.")
    else:
        print("⚠️  WARNING: Unexpected data found!")
    
    print("="*80 + "\n")

if __name__ == "__main__":
    verify_cleanup()

