"""
Update Track Order Prompt in Database
Improves Arabic number handling and intent detection
"""
import asyncio
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from supabase import create_client
from app.config import settings
from app.utils.logger import app_logger as logger


async def update_track_order_prompt():
    """Update the track_order prompt section in database"""
    
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    # New improved prompt content
    new_content = """## 🔍 Intent: Track Order

**When to use**: Customer wants to check order status

**Trigger phrases**:
- Arabic: "وين طلبي", "ما حالة طلبي", "ما حالة الطلب", "طلبي وصل", "كم باقي على طلبي", "متى يوصل طلبي", "ما اخر اوردر", "آخر طلب", "اخر طلب", "آخر أوردر", "اخر اوردر قمت بطلبه", "ما اخر طلب", "اخر طلب خاص بي"
- English: "where is my order", "order status", "what's the status", "track order", "check order", "last order", "latest order", "my last order", "my latest order"

**🚨 CRITICAL RULES:**
1. **NEVER** say "ليس لديك طلب" or "You don't have an order" WITHOUT calling track_order action first!
2. **NEVER** say "لإلغائه" (to cancel it) when customer asks about order STATUS - that's for CANCEL intent!
3. **ALWAYS** use `action: "track_order"` when customer asks about order status
4. **DO NOT** rely on draft_sessions to determine if customer has orders - they may have completed orders in the database!
5. The draft_session only shows INCOMPLETE orders. Customer may have COMPLETED orders that you MUST search for!

**🔴 DISTINGUISH FROM CANCEL ORDER:**
- Track Order: "ما حالة", "وين", "آخر طلب" → Customer wants to CHECK status
- Cancel Order: "ألغي", "إلغاء", "لا أريد" → Customer wants to CANCEL
- **NEVER** mix these two intents!

**Important Notes**:
1. ⚠️ **Order Number Extraction**:
   - The system automatically extracts order numbers from messages
   - If you see "EXTRACTED ORDER NUMBER: XXXXX" in the context, USE IT
   - Order numbers can be in formats: "10001", "2025-019", "019"
   - Arabic numerals (٠-٩) are automatically converted to English (0-9)

2. **Response Strategy**:
   - If order number is extracted → Use it immediately for tracking with `action: "track_order"`
   - If customer asks for "last order" or "latest order" → **MUST** use `action: "track_order"` with `data: null`
   - If no order number and not asking for latest → Ask customer to provide it (but still use `action: null`)
   - Always respond in the same language as customer's message

3. **Action Required**:
```json
{
  "intent": "track_order",
  "reply": "[Your response in customer's language]",
  "action": "track_order",
  "data": {
    "order_number": "[extracted order number OR null for latest order]"
  }
}
```

**Example Scenarios**:

Scenario 1: Arabic with Arabic numerals
```
Customer: "ما حالة طلبي رقم ١٠٠٠١"
System extracts: "10001"
Response: {
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك رقم 10001...",
  "action": "track_order",
  "data": {"order_number": "10001"}
}
```

Scenario 2: Customer asks for latest order ⚠️ IMPORTANT!
```
Customer: "ما اخر اوردر قمت بطلبه"
No specific order number
🚨 MUST use action: "track_order" with data: null
Response: {
  "intent": "track_order",
  "reply": "جاري البحث عن آخر طلب لك...",
  "action": "track_order",
  "data": null
}

❌ WRONG Response #1 (DO NOT DO THIS):
{
  "intent": "track_order",
  "reply": "ليس لديك طلب حالياً",
  "action": null,
  "data": null
}

❌ WRONG Response #2 (DO NOT DO THIS - This is for CANCEL, not TRACK):
{
  "intent": "track_order",
  "reply": "ليس لديك طلب حالياً لإلغائه",
  "action": null,
  "data": null
}
```

Scenario 3: Customer sends only number (CHECK CONVERSATION HISTORY!)
```
Customer: "10001"
Context: Check recent conversation history!
- If previous message was about tracking → Intent is "track_order"
- If previous message was about canceling → Intent is "cancel_order"
- If no context → Ask what they want to do

Response (if previous was tracking): {
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك رقم 10001...",
  "action": "track_order",
  "data": {"order_number": "10001"}
}
```

Scenario 4: No order number and not asking for latest
```
Customer: "وين طلبي؟"
No extracted number, not asking for latest
Response: {
  "intent": "track_order",
  "reply": "من فضلك أعطني رقم الطلب لأتمكن من البحث عنه.",
  "action": null,
  "data": null
}
```

**Status Messages** (based on order status from database):
- `pending`: "طلبك قيد التحضير حالياً 👨‍🍳"
- `confirmed`: "تم تأكيد طلبك وجاري التحضير 🔥"
- `preparing`: "طلبك يتم تحضيره الآن 🍳"
- `ready`: "طلبك جاهز! 🎉"
- `out_for_delivery`: "طلبك في الطريق إليك 🚗"
- `delivered`: "تم توصيل طلبك بنجاح ✅"
- `cancelled`: "تم إلغاء هذا الطلب ❌"

**Error Handling**:
- If order not found: "عذراً، لم أجد طلب بهذا الرقم. تأكد من الرقم وحاول مرة أخرى."
- If order belongs to different customer: "هذا الطلب غير مرتبط بحسابك."
"""

    try:
        # Update the prompt section
        response = supabase.table('prompt_sections') \
            .update({'content': new_content}) \
            .eq('section_key', 'intent_track_order') \
            .execute()
        
        if response.data:
            logger.info("✅ Successfully updated intent_track_order prompt")
            print("✅ Track Order prompt updated successfully!")
            print(f"Updated {len(response.data)} row(s)")
        else:
            logger.warning("⚠️ No rows updated - section might not exist")
            print("⚠️ No rows updated. Creating new section...")
            
            # Create new section if it doesn't exist
            new_section = {
                'section_key': 'intent_track_order',
                'section_name': 'Track Order Intent',
                'content': new_content,
                'priority': 60,
                'is_required': False,
                'active': True
            }
            
            response = supabase.table('prompt_sections').insert(new_section).execute()
            
            if response.data:
                logger.info("✅ Created new intent_track_order section")
                print("✅ New section created successfully!")
            else:
                logger.error("❌ Failed to create section")
                print("❌ Failed to create section")
        
    except Exception as e:
        logger.error(f"Error updating prompt: {e}")
        print(f"❌ Error: {e}")
        raise


if __name__ == "__main__":
    print("🔄 Updating Track Order prompt in database...")
    asyncio.run(update_track_order_prompt())
    print("✅ Done!")

