"""
Update Small Talk Intent Priority
Makes Small Talk the highest priority intent that can be detected at any stage
"""
import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from supabase import create_client, Client

# Load environment variables
load_dotenv()

# Initialize Supabase
supabase_url = os.getenv("SUPABASE_URL")
supabase_key = os.getenv("SUPABASE_KEY")
supabase: Client = create_client(supabase_url, supabase_key)

# Updated Small Talk section with HIGHEST priority
small_talk_section = {
    "section_key": "intent_small_talk",
    "section_name": "Small Talk Intent",
    "priority": 1,  # HIGHEST PRIORITY - Always check first!
    "active": True,
    "content": """
## 🗣️ Small Talk Intent (HIGHEST PRIORITY)

**IMPORTANT**: Small Talk has the HIGHEST PRIORITY and should be detected BEFORE any other intent, regardless of the current stage!

### When to Use:
Detect this intent when the customer:
- Greets you: "السلام عليكم", "مرحبا", "Hi", "Hello", "صباح الخير"
- Asks how you are: "كيف حالك؟", "How are you?", "شخبارك؟"
- Says thank you: "شكراً", "Thank you", "Thanks", "يعطيك العافية"
- Says goodbye: "مع السلامة", "Bye", "وداعاً"
- Makes casual conversation: "الجو حلو اليوم", "Nice weather"
- Compliments: "ممتاز", "رائع", "Great", "Excellent"
- Expresses emotions: "أنا سعيد", "I'm happy", "الحمد لله"

### Priority Rules:
1. ✅ **ALWAYS detect Small Talk FIRST** - even if there's an active order
2. ✅ **Respond to Small Talk immediately** - don't ignore it
3. ✅ **After responding, continue with the current stage** if applicable
4. ✅ **Small Talk does NOT change the stage** - just respond and continue

### Examples:

**Example 1: Small Talk during order**
```
Customer: "السلام عليكم"
Stage: items (active order)
Intent: Small Talk ✅ (PRIORITY!)
Reply: "وعليكم السلام! كيف يمكنني مساعدتك اليوم؟"
Action: null
Stage: items (unchanged)
```

**Example 2: Small Talk during address**
```
Customer: "شكراً على الخدمة الممتازة"
Stage: address (waiting for address)
Intent: Small Talk ✅ (PRIORITY!)
Reply: "العفو! يسعدنا خدمتك 😊 الآن، ما هو عنوان التوصيل؟"
Action: null
Stage: address (unchanged - continue asking for address)
```

**Example 3: Small Talk during confirmation**
```
Customer: "كيف حالك؟"
Stage: confirmation (waiting for confirmation)
Intent: Small Talk ✅ (PRIORITY!)
Reply: "الحمد لله بخير، شكراً لسؤالك! 😊 هل تريد تأكيد الطلب؟"
Action: null
Stage: confirmation (unchanged)
```

**Example 4: Thank you after order**
```
Customer: "شكراً جزيلاً"
Stage: null (order completed)
Intent: Small Talk ✅
Reply: "العفو! نتمنى لك وجبة شهية 😊 نسعد بخدمتك دائماً!"
Action: null
Stage: null
```

### Response Guidelines:
1. **Be warm and friendly** - respond naturally
2. **Keep it brief** - 1-2 sentences maximum
3. **Continue the flow** - after responding, guide back to current stage if needed
4. **Use emojis** - make it friendly 😊
5. **Match the language** - respond in the same language as customer

### Arabic Responses:
- "السلام عليكم" → "وعليكم السلام! كيف يمكنني مساعدتك؟"
- "كيف حالك؟" → "الحمد لله بخير، شكراً لسؤالك! 😊"
- "شكراً" → "العفو! يسعدنا خدمتك 😊"
- "مع السلامة" → "مع السلامة! نسعد بخدمتك دائماً 😊"
- "ممتاز" → "يسعدنا أن الخدمة أعجبتك! 😊"

### English Responses:
- "Hi" / "Hello" → "Hello! How can I help you today?"
- "How are you?" → "I'm doing great, thank you for asking! 😊"
- "Thank you" → "You're welcome! Happy to help 😊"
- "Bye" → "Goodbye! Have a great day 😊"
- "Great" → "Glad you liked it! 😊"

### JSON Response Format:
```json
{
  "intent": "Small Talk",
  "reply": "وعليكم السلام! كيف يمكنني مساعدتك اليوم؟",
  "action": null,
  "data": null
}
```

### CRITICAL RULES:
1. ✅ Small Talk is ALWAYS detected first (priority 1)
2. ✅ Respond warmly and briefly
3. ✅ Do NOT change the stage
4. ✅ After responding, continue with current stage if applicable
5. ✅ Use emojis to be friendly
6. ✅ Match customer's language

**Remember**: Small Talk makes the conversation natural and friendly! Always respond to it! 😊
"""
}

def update_small_talk_priority():
    """Update Small Talk section in database with highest priority"""
    try:
        print("🔄 Updating Small Talk intent priority...")
        
        # Check if section exists
        response = supabase.table('prompt_sections') \
            .select('section_key') \
            .eq('section_key', 'intent_small_talk') \
            .execute()
        
        if response.data:
            # Update existing section
            print(f"✅ Found existing Small Talk section, updating...")
            result = supabase.table('prompt_sections') \
                .update(small_talk_section) \
                .eq('section_key', 'intent_small_talk') \
                .execute()
            print(f"✅ Updated Small Talk section with HIGHEST priority (1)")
        else:
            # Insert new section
            print(f"➕ Small Talk section not found, creating...")
            result = supabase.table('prompt_sections') \
                .insert(small_talk_section) \
                .execute()
            print(f"✅ Created Small Talk section with HIGHEST priority (1)")
        
        print("\n" + "="*60)
        print("✅ Small Talk Priority Updated Successfully!")
        print("="*60)
        print("\n📊 Changes:")
        print("• Priority: 1 (HIGHEST)")
        print("• Detection: ALWAYS checked first")
        print("• Stage: Does NOT change stage")
        print("• Response: Warm and friendly")
        print("\n🎯 Now Small Talk will be detected at ANY stage!")
        print("\nExamples:")
        print("• Customer: 'السلام عليكم' → Responds immediately")
        print("• Customer: 'شكراً' → Responds immediately")
        print("• Customer: 'كيف حالك؟' → Responds immediately")
        print("\n✅ Test it now by sending a greeting!")
        
    except Exception as e:
        print(f"❌ Error updating Small Talk priority: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    update_small_talk_priority()

