#!/usr/bin/env python3
"""
Script to update menu_items table schema and add descriptions/tags
"""
from supabase import create_client
import sys

# Supabase credentials
SUPABASE_URL = "https://yoeyovuanxrsdfgokfgh.supabase.co"
SUPABASE_SERVICE_KEY = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InlvZXlvdnVhbnhyc2RmZ29rZmdoIiwicm9sZSI6InNlcnZpY2Vfcm9sZSIsImlhdCI6MTc1NzU5NjMwMywiZXhwIjoyMDczMTcyMzAzfQ.RTTcsH5DHpElzwXCH-OPAVemW-yxgLF3VH0iBF9qkF4"

supabase = create_client(SUPABASE_URL, SUPABASE_SERVICE_KEY)

print("🚀 Starting menu_items table enhancement...")
print("="*100)

# Read SQL migration file
try:
    with open('database/migrations/002_enhance_menu_items.sql', 'r', encoding='utf-8') as f:
        sql_content = f.read()
    
    print("✅ Loaded SQL migration file")
    
    # Split SQL into individual statements
    statements = [s.strip() for s in sql_content.split(';') if s.strip() and not s.strip().startswith('--')]
    
    print(f"📝 Found {len(statements)} SQL statements to execute")
    print()
    
    # Execute each statement
    success_count = 0
    error_count = 0
    
    for i, statement in enumerate(statements, 1):
        # Skip comments and empty statements
        if not statement or statement.startswith('--'):
            continue
            
        try:
            # Show what we're executing (first 100 chars)
            preview = statement[:100].replace('\n', ' ')
            print(f"[{i}/{len(statements)}] Executing: {preview}...")
            
            # Execute via Supabase
            result = supabase.postgrest.rpc('exec_sql', {'query': statement}).execute()
            
            print(f"  ✅ Success")
            success_count += 1
            
        except Exception as e:
            error_msg = str(e)
            
            # Check if it's a "column already exists" error (which is OK)
            if 'already exists' in error_msg.lower() or 'duplicate' in error_msg.lower():
                print(f"  ⚠️  Already exists (skipping)")
                success_count += 1
            else:
                print(f"  ❌ Error: {error_msg[:100]}")
                error_count += 1
    
    print()
    print("="*100)
    print(f"📊 Results: {success_count} successful, {error_count} errors")
    
    if error_count > 0:
        print("\n⚠️  Some statements failed. Trying alternative approach...")
        print("="*100)
        
        # Alternative: Update items one by one using Supabase client
        print("\n🔄 Updating menu items with descriptions and tags...")
        
        updates = [
            # أطباق عمانية
            {'code': 'HAREES_CHICKEN', 'description': 'هريس تقليدي عماني بالدجاج، طبق شعبي محبوب', 'tags': ['هريس', 'دجاج', 'أطباق عمانية', 'تقليدي', 'شعبي']},
            {'code': 'HAREES_MEAT', 'description': 'هريس تقليدي عماني باللحم، غني بالبروتين', 'tags': ['هريس', 'لحم', 'أطباق عمانية', 'تقليدي', 'بروتين']},
            {'code': 'MASLOONA_CHICKEN', 'description': 'مصلونة دجاج مع رز أبيض، وجبة متكاملة', 'tags': ['مصلونة', 'دجاج', 'رز', 'أطباق عمانية', 'وجبة كاملة']},
            {'code': 'MASLOONA_MEAT', 'description': 'مصلونة لحم مع رز أبيض، طعم غني ولذيذ', 'tags': ['مصلونة', 'لحم', 'رز', 'أطباق عمانية', 'وجبة كاملة']},
            {'code': 'MASLOONA_TUNA', 'description': 'مصلونة سمك تونة مع رز أبيض، خيار صحي', 'tags': ['مصلونة', 'سمك', 'تونة', 'رز', 'أطباق عمانية', 'صحي']},
            {'code': 'ARSI_JALAN', 'description': 'عرسي جعلان بلحم الغنم المحلي، طبق فاخر', 'tags': ['عرسي', 'جعلان', 'غنم', 'لحم محلي', 'أطباق عمانية', 'فاخر']},
            {'code': 'MASORA_AWAL', 'description': 'مصورة أول مع رز أبيض، نكهة مميزة', 'tags': ['مصورة', 'أول', 'رز', 'أطباق عمانية']},
            {'code': 'KABSA_AWAN', 'description': 'كبسة عوان، طبق عماني أصيل', 'tags': ['كبسة', 'عوان', 'رز', 'أطباق عمانية', 'أصيل']},
            {'code': 'MALEH_RICE', 'description': 'مالح مع رز أبيض، طعم مالح مميز', 'tags': ['مالح', 'رز', 'أطباق عمانية']},
            {'code': 'SAHNA_RICE', 'description': 'صحنة مع رز أبيض، وجبة خفيفة', 'tags': ['صحنة', 'رز', 'أطباق عمانية', 'خفيف']},
            {'code': 'QASHAA_BARIA', 'description': 'قاشع برية مع ليمون ورز أبيض، نكهة منعشة', 'tags': ['قاشع', 'برية', 'ليمون', 'رز', 'أطباق عمانية', 'منعش']},
            {'code': 'FISH_FRIED', 'description': 'سمك مقلي مع رز قبولي، مقرمش ولذيذ', 'tags': ['سمك', 'مقلي', 'رز قبولي', 'أطباق عمانية', 'مقرمش']},
            {'code': 'CAMEL_MEAT', 'description': 'لحم جمل مع رز أبيض، طعم فريد', 'tags': ['لحم', 'جمل', 'رز', 'أطباق عمانية', 'فريد']},
            {'code': 'CAMEL_QASABIA', 'description': 'لحم جمل قصابية، تحضير خاص', 'tags': ['لحم', 'جمل', 'قصابية', 'أطباق عمانية', 'خاص']},
            
            # شوربات
            {'code': 'SOUP_LENTIL', 'description': 'شوربة عدس ساخنة، غنية بالبروتين', 'tags': ['شوربة', 'عدس', 'ساخن', 'بروتين', 'صحي']},
            {'code': 'SOUP_HAREES', 'description': 'شوربة هريس باللحم، دافئة ومغذية', 'tags': ['شوربة', 'هريس', 'لحم', 'دافئ', 'مغذي']},
            {'code': 'SOUP_CORN', 'description': 'شوربة ذرة بالدجاج، كريمية ولذيذة', 'tags': ['شوربة', 'ذرة', 'دجاج', 'كريمي', 'لذيذ']},
            {'code': 'SOUP_SHRIMP', 'description': 'شوربة بايبلو بالروبيان، نكهة بحرية', 'tags': ['شوربة', 'بايبلو', 'روبيان', 'بحري', 'فاخر']},
            {'code': 'SOUP_MUSHROOM', 'description': 'شوربة مشروم بالكريمة، غنية وكريمية', 'tags': ['شوربة', 'مشروم', 'كريمة', 'غني', 'كريمي']},
            
            # سلطات
            {'code': 'SALAD_TABBOULEH', 'description': 'تبولة طازجة، سلطة لبنانية كلاسيكية', 'tags': ['سلطة', 'تبولة', 'طازج', 'لبناني', 'صحي']},
            {'code': 'SALAD_ARABIC', 'description': 'سلطة عربية مشكلة، خضار طازجة', 'tags': ['سلطة', 'عربي', 'خضار', 'طازج', 'صحي']},
            {'code': 'SALAD_ROCCA', 'description': 'سلطة جرجير، غنية بالفيتامينات', 'tags': ['سلطة', 'جرجير', 'فيتامينات', 'صحي', 'طازج']},
            {'code': 'SALAD_GREEK', 'description': 'سلطة يونانية بجبن الفيتا، نكهة متوسطية', 'tags': ['سلطة', 'يوناني', 'فيتا', 'جبن', 'متوسطي']},
            {'code': 'SALAD_FATTOUSH', 'description': 'فتوش، سلطة شامية مع خبز محمص', 'tags': ['سلطة', 'فتوش', 'شامي', 'خبز محمص', 'طازج']},
            
            # مقبلات
            {'code': 'APP_KIBBEH', 'description': 'كبة مقلية، مقبلات لبنانية مقرمشة', 'tags': ['مقبلات', 'كبة', 'مقلي', 'مقرمش', 'لبناني']},
            {'code': 'APP_SAMBOUSEK', 'description': 'سمبوسة باللحم، مقبلات محشية', 'tags': ['مقبلات', 'سمبوسة', 'لحم', 'محشي', 'مقرمش']},
            {'code': 'APP_HUMMUS', 'description': 'حمص، مقبلات شرقية كلاسيكية', 'tags': ['مقبلات', 'حمص', 'شرقي', 'كلاسيكي', 'صحي']},
            {'code': 'APP_MOUTABAL', 'description': 'متبل، باذنجان مشوي مع طحينة', 'tags': ['مقبلات', 'متبل', 'باذنجان', 'طحينة', 'شرقي']},
            {'code': 'APP_VINE_LEAVES', 'description': 'ورق عنب، محشي بالأرز واللحم', 'tags': ['مقبلات', 'ورق عنب', 'محشي', 'أرز', 'لحم']},
            
            # مشروبات
            {'code': 'BEV_SOFT_DRINK', 'description': 'مشروب غازي بارد، منعش', 'tags': ['مشروبات', 'غازي', 'بارد', 'منعش']},
            {'code': 'BEV_JUICE_FRESH', 'description': 'عصير طازج، فواكه طبيعية', 'tags': ['مشروبات', 'عصير', 'طازج', 'طبيعي', 'فواكه']},
            {'code': 'BEV_LABAN', 'description': 'لبن طازج، مشروب تقليدي', 'tags': ['مشروبات', 'لبن', 'طازج', 'تقليدي']},
            {'code': 'BEV_WATER', 'description': 'ماء معدني، نقي ومنعش', 'tags': ['مشروبات', 'ماء', 'معدني', 'نقي', 'منعش']},
            
            # عروض خاصة
            {'code': 'OFFER_C6889546', 'description': 'عرض الغداء الخاص: 3 هريس دجاج + 3 مضروبة لحم + سلطة مجاناً', 'tags': ['عروض', 'غداء', 'هريس', 'مضروبة', 'وجبة عائلية']},
            {'code': 'OFFER_BF065598', 'description': 'عرض الوجبة العائلية: 5 هريس لحم + 5 كبسة عوان + 2 لتر بيبسي', 'tags': ['عروض', 'عائلي', 'هريس', 'كبسة', 'وجبة كبيرة']},
            
            # تحديث الأصناف بدون فئة
            {'code': 'COLA_SM', 'category': 'مشروبات', 'description': 'مشروب غازي صغير، حجم مناسب', 'tags': ['مشروبات', 'غازي', 'صغير', 'كولا']},
            {'code': 'BURGER_CL', 'category': 'وجبات سريعة', 'description': 'برجر كلاسيك، لحم طازج', 'tags': ['برجر', 'لحم', 'وجبة سريعة', 'كلاسيكي']},
            {'code': 'FRIES_MD', 'category': 'وجبات سريعة', 'description': 'بطاطس مقلية متوسطة، مقرمشة', 'tags': ['بطاطس', 'مقلي', 'وجبة سريعة', 'مقرمش']},
        ]
        
        updated_count = 0
        failed_count = 0
        
        for update_data in updates:
            code = update_data.pop('code')
            try:
                result = supabase.table('menu_items').update(update_data).eq('code', code).execute()
                print(f"  ✅ Updated {code}")
                updated_count += 1
            except Exception as e:
                print(f"  ❌ Failed to update {code}: {str(e)[:50]}")
                failed_count += 1
        
        print()
        print("="*100)
        print(f"📊 Update Results: {updated_count} successful, {failed_count} failed")
    
    # Verify the changes
    print("\n🔍 Verifying changes...")
    print("="*100)
    
    result = supabase.table('menu_items').select('code, name_ar, description, tags').limit(5).execute()
    
    if result.data:
        print("\n✅ Sample of updated items:")
        for item in result.data:
            print(f"\nCode: {item['code']}")
            print(f"  Name: {item['name_ar']}")
            print(f"  Description: {item.get('description', 'N/A')}")
            print(f"  Tags: {item.get('tags', 'N/A')}")
    
    print("\n" + "="*100)
    print("✅ Menu enhancement completed successfully!")
    print("\n📝 Next steps:")
    print("  1. Verify the changes in Supabase dashboard")
    print("  2. Test the search functionality with new tags")
    print("  3. Update FAISS embeddings if needed")
    
except FileNotFoundError:
    print("❌ Error: Could not find migration file")
    print("   Make sure you're running this from the restaurant-chatbot-server directory")
    sys.exit(1)
except Exception as e:
    print(f"❌ Error: {str(e)}")
    sys.exit(1)

