#!/usr/bin/env python3
"""
Update Menu Cache Script
========================
Updates the local menu cache (menu_cache.json) from Supabase database.

Run this script after any menu updates in Supabase to refresh the cache.

Usage:
    python scripts/update_menu_cache.py
"""

import json
import os
import sys
from datetime import datetime
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.database.supabase_client import get_supabase_client
from app.utils.logger import app_logger as logger


def update_menu_cache():
    """
    Fetch menu items from Supabase and update local cache
    """
    try:
        print("=" * 80)
        print("🔄 Updating Menu Cache from Supabase")
        print("=" * 80)
        
        # 1. Connect to Supabase
        print("\n1️⃣ Connecting to Supabase...")
        supabase = get_supabase_client()
        
        # 2. Fetch all active menu items
        print("2️⃣ Fetching menu items...")
        menu_items = supabase.get_all_menu_items(active_only=True)
        print(f"✅ Fetched {len(menu_items)} active menu items")
        
        # 3. Prepare cache data
        print("\n3️⃣ Preparing cache data...")
        cache_data = {
            'items': menu_items,
            'count': len(menu_items),
            'timestamp': datetime.now().isoformat(),
            'source': 'supabase'
        }
        
        # 4. Save to cache file
        print("4️⃣ Saving to cache file...")
        cache_dir = Path(__file__).parent.parent / 'data' / 'cache'
        cache_dir.mkdir(parents=True, exist_ok=True)
        
        cache_file = cache_dir / 'menu_cache.json'
        with open(cache_file, 'w', encoding='utf-8') as f:
            json.dump(cache_data, f, ensure_ascii=False, indent=2)
        
        print(f"✅ Cache saved to: {cache_file}")
        
        # 5. Save metadata
        print("5️⃣ Saving metadata...")
        metadata = {
            'last_update': datetime.now().isoformat(),
            'items_count': len(menu_items),
            'categories': list(set(item.get('category', 'Unknown') for item in menu_items))
        }
        
        metadata_file = cache_dir / 'menu_metadata.json'
        with open(metadata_file, 'w', encoding='utf-8') as f:
            json.dump(metadata, f, ensure_ascii=False, indent=2)
        
        print(f"✅ Metadata saved to: {metadata_file}")
        
        # 6. Summary
        print("\n" + "=" * 80)
        print("✅ Menu Cache Updated Successfully!")
        print("=" * 80)
        print(f"📊 Summary:")
        print(f"   • Total items: {len(menu_items)}")
        print(f"   • Categories: {len(metadata['categories'])}")
        print(f"   • Updated at: {metadata['last_update']}")
        print(f"   • Cache file: {cache_file}")
        print("\n💡 Restart the server to load the new cache:")
        print("   pkill -f uvicorn && python -m uvicorn app.main:app --reload")
        print("=" * 80)
        
        return True
        
    except Exception as e:
        print(f"\n❌ Error updating menu cache: {str(e)}")
        import traceback
        traceback.print_exc()
        return False


if __name__ == "__main__":
    success = update_menu_cache()
    sys.exit(0 if success else 1)

