#!/usr/bin/env python3
"""
Test Word Correction Dictionary
اختبار قاموس تصحيح الكلمات
"""

import os
import sys

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.utils.word_correction_dict import word_correction_dict

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*70)
    print(f"  {text}")
    print("="*70)

def test_single_word_correction():
    """Test correct_word function"""
    
    print_header("🧪 Testing Single Word Correction")
    
    # Test cases: (wrong_word, expected_correct_word)
    test_cases = [
        # أخطاء إملائية عربية
        ("سمج", "سمك"),
        ("صمك", "سمك"),
        ("دجاچ", "دجاج"),
        ("هريص", "هريس"),
        ("كبصة", "كبسة"),
        ("برقر", "برجر"),
        ("شورما", "شاورما"),
        ("حمض", "حمص"),
        ("فتوچ", "فتوش"),
        ("تبوله", "تبولة"),
        ("عصر", "عصير"),
        ("قهوه", "قهوة"),
        ("رس", "رز"),
        ("مقلى", "مقلي"),
        ("مشوى", "مشوي"),
        
        # لهجات
        ("ابي", "أريد"),
        ("ابغى", "أريد"),
        ("بدي", "أريد"),
        ("عايز", "أريد"),
        ("ايش", "ما"),
        ("شو", "ما"),
        ("وش", "ما"),
        ("بكم", "كم"),
        ("بكام", "كم"),
        ("ايوه", "نعم"),
        ("زين", "نعم"),
        
        # أخطاء إملائية إنجليزية
        ("fich", "fish"),
        ("chiken", "chicken"),
        ("burgr", "burger"),
        ("shawerma", "shawarma"),
        ("shremp", "shrimp"),
        ("rize", "rice"),
        ("griled", "grilled"),
        ("fryed", "fried"),
        
        # كلمات صحيحة (لا تصحيح)
        ("سمك", "سمك"),
        ("دجاج", "دجاج"),
        ("هريس", "هريس"),
        ("fish", "fish"),
        ("chicken", "chicken"),
    ]
    
    passed = 0
    failed = 0
    
    for wrong, expected in test_cases:
        corrected = word_correction_dict.correct_word(wrong)
        
        if corrected == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{wrong}' → '{corrected}' (expected: '{expected}')")
    
    # Summary
    print_header("📊 Single Word Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"❌ Failed: {failed}/{len(test_cases)}")
    print(f"Success Rate: {passed/len(test_cases)*100:.1f}%")
    
    return passed, failed

def test_text_correction():
    """Test correct_text function"""
    
    print_header("🧪 Testing Full Text Correction")
    
    # Test cases: (wrong_text, min_corrections, max_corrections)
    # Some corrections may vary due to fuzzy matching
    test_cases = [
        ("سمج مقلى", 2, 2),  # سمج → سمك, مقلى → مقلي
        ("ابي دجاچ مشوى", 3, 3),  # ابي → أريد, دجاچ → دجاج, مشوى → مشوي
        ("عندك هريص؟", 1, 2),  # هريص → هريس (عندك → عندكم optional)
        ("بكم كبصة عوان؟", 2, 2),  # بكم → كم, كبصة → كبسة
        ("ايش عندكم من برقر؟", 2, 2),  # ايش → ما, برقر → برجر
        ("بدي شورما دجاچ", 3, 3),  # بدي → أريد, شورما → شاورما, دجاچ → دجاج
        ("fich and chiken", 2, 2),  # fich → fish, chiken → chicken
        ("I want burgr please", 1, 2),  # burgr → burger (I might be corrected)
        ("سمك مقلي", 0, 0),  # لا تصحيحات (كل شيء صحيح)
        ("fish and chicken", 0, 0),  # لا تصحيحات
    ]
    
    passed = 0
    failed = 0

    for text, min_count, max_count in test_cases:
        corrected_text, corrections = word_correction_dict.correct_text(text)
        actual_count = len(corrections)

        if min_count <= actual_count <= max_count:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1

        print(f"{icon} '{text}'")
        print(f"   → '{corrected_text}'")
        print(f"   Corrections: {actual_count} (expected: {min_count}-{max_count})")
        if corrections:
            for corr in corrections:
                print(f"      • '{corr['original']}' → '{corr['corrected']}'")
        print()
    
    # Summary
    print_header("📊 Full Text Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"❌ Failed: {failed}/{len(test_cases)}")
    print(f"Success Rate: {passed/len(test_cases)*100:.1f}%")
    
    return passed, failed

def test_fuzzy_matching():
    """Test fuzzy matching for typos"""
    
    print_header("🧪 Testing Fuzzy Matching")
    
    # Test cases: words with small typos that should match menu items
    test_cases = [
        "سمگ",  # should match سمك
        "دجاچ",  # should match دجاج
        "هرس",  # should match هريس
        "كبسه",  # should match كبسة
    ]
    
    passed = 0
    failed = 0
    
    for word in test_cases:
        corrected = word_correction_dict.correct_word(word)
        
        # Check if correction happened
        if corrected != word.lower():
            icon = "✅"
            passed += 1
            print(f"{icon} '{word}' → '{corrected}' (fuzzy match)")
        else:
            icon = "⚠️"
            failed += 1
            print(f"{icon} '{word}' → no correction (fuzzy match failed)")
    
    # Summary
    print_header("📊 Fuzzy Matching Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"⚠️  Failed: {failed}/{len(test_cases)}")
    print(f"Success Rate: {passed/len(test_cases)*100:.1f}%")
    
    return passed, failed

def main():
    """Run all tests"""
    
    print_header("🚀 Word Correction Dictionary Tests")
    print(f"Dictionary size: {word_correction_dict.get_corrections_count()} entries")
    print(f"Menu keywords: {len(word_correction_dict.menu_keywords)} words")
    
    # Run tests
    single_passed, single_failed = test_single_word_correction()
    text_passed, text_failed = test_text_correction()
    fuzzy_passed, fuzzy_failed = test_fuzzy_matching()
    
    # Overall summary
    total_passed = single_passed + text_passed + fuzzy_passed
    total_failed = single_failed + text_failed + fuzzy_failed
    total_tests = total_passed + total_failed
    
    print_header("🎯 Overall Summary")
    print(f"✅ Total Passed: {total_passed}/{total_tests}")
    print(f"❌ Total Failed: {total_failed}/{total_tests}")
    print(f"Success Rate: {total_passed/total_tests*100:.1f}%")
    
    if total_failed == 0:
        print("\n🎉 All tests passed!")
    else:
        print(f"\n⚠️  {total_failed} test(s) failed")

if __name__ == "__main__":
    main()

