#!/usr/bin/env python3
"""
Test Unified Menu Search Service
اختبار خدمة البحث الموحد في المنيو
"""

import os
import sys
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.services.menu_search_service import menu_search_service

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

def print_results(search_result: dict):
    """Print search results in a formatted way"""
    results = search_result.get('results', [])
    search_methods = search_result.get('search_methods', [])
    corrected_query = search_result.get('corrected_query', '')
    total_found = search_result.get('total_found', 0)
    
    print(f"\n📊 Search Summary:")
    print(f"   Query: '{corrected_query}'")
    print(f"   Methods: {', '.join(search_methods)}")
    print(f"   Total: {total_found} items")
    print(f"   Exact: {search_result.get('exact_count', 0)}")
    print(f"   Partial: {search_result.get('partial_count', 0)}")
    print(f"   Semantic: {search_result.get('semantic_count', 0)}")
    
    if not results:
        print("\n❌ No results found")
        return
    
    print(f"\n📋 Results:")
    for i, item in enumerate(results, 1):
        name_ar = item.get('name_ar', 'N/A')
        name_en = item.get('name_en', 'N/A')
        price = item.get('price', 0)
        match_type = item.get('match_type', 'unknown')
        match_score = item.get('match_score', 0)
        
        # Icon based on match type
        if match_type == 'exact':
            icon = '🎯'
        elif match_type == 'partial':
            icon = '🔍'
        else:
            icon = '🧠'
        
        print(f"{icon} {i}. {name_ar} / {name_en}")
        print(f"   💰 {price} ريال | {match_type} ({match_score:.2f})")

def test_exact_match():
    """Test exact match search"""
    print_header("🧪 Test 1: Exact Match Search")
    
    test_cases = [
        "سمك",
        "fish",
        "هريس",
        "harees",
        "كبسة",
        "kabsa"
    ]
    
    for query in test_cases:
        print(f"\n🔍 Searching for: '{query}'")
        result = menu_search_service.search_menu(query)
        
        if result['exact_count'] > 0:
            print(f"✅ Exact match found!")
        else:
            print(f"⚠️  No exact match")
        
        print_results(result)

def test_partial_match():
    """Test partial match search"""
    print_header("🧪 Test 2: Partial Match Search")
    
    test_cases = [
        "شوربة",  # should find "شوربة هريس"
        "مقلي",   # should find items with "مقلي"
        "مشوي",   # should find items with "مشوي"
        "دجاج",   # should find items with "دجاج"
        "لحم",    # should find items with "لحم"
    ]
    
    for query in test_cases:
        print(f"\n🔍 Searching for: '{query}'")
        result = menu_search_service.search_menu(query)
        
        if result['partial_count'] > 0:
            print(f"✅ Partial match found!")
        else:
            print(f"⚠️  No partial match")
        
        print_results(result)

def test_semantic_search():
    """Test semantic search"""
    print_header("🧪 Test 3: Semantic Search")
    
    test_cases = [
        "أكل بحري",      # should find fish items
        "وجبة خفيفة",    # should find appetizers
        "حلويات",        # should find desserts
        "مشروبات باردة", # should find cold drinks
    ]
    
    for query in test_cases:
        print(f"\n🔍 Searching for: '{query}'")
        result = menu_search_service.search_menu(query)
        
        if result['semantic_count'] > 0:
            print(f"✅ Semantic match found!")
        else:
            print(f"⚠️  No semantic match")
        
        print_results(result)

def test_word_correction():
    """Test word correction integration"""
    print_header("🧪 Test 4: Word Correction Integration")
    
    test_cases = [
        ("سمج", "سمك"),           # spelling mistake
        ("دجاچ", "دجاج"),         # spelling mistake
        ("هريص", "هريس"),         # spelling mistake
        ("ابي سمك", "أريد سمك"),  # dialect
        ("بدي دجاج", "أريد دجاج"), # dialect
    ]
    
    for wrong, expected in test_cases:
        print(f"\n🔍 Searching for: '{wrong}' (expected: '{expected}')")
        result = menu_search_service.search_menu(wrong)
        
        corrected = result.get('corrected_query', wrong)
        
        if corrected != wrong:
            print(f"✅ Corrected: '{wrong}' → '{corrected}'")
        else:
            print(f"⚠️  No correction applied")
        
        print_results(result)

def test_priority_ranking():
    """Test priority ranking (Exact > Partial > Semantic)"""
    print_header("🧪 Test 5: Priority Ranking")
    
    # Query that should trigger all 3 methods
    query = "سمك"
    
    print(f"\n🔍 Searching for: '{query}'")
    result = menu_search_service.search_menu(query, limit=20)
    
    print_results(result)
    
    # Verify ranking
    results = result.get('results', [])
    if results:
        print(f"\n📊 Ranking Verification:")
        
        # Check if exact matches come first
        exact_items = [r for r in results if r.get('match_type') == 'exact']
        partial_items = [r for r in results if r.get('match_type') == 'partial']
        semantic_items = [r for r in results if r.get('match_type') == 'semantic']
        
        print(f"   🥇 Exact matches: {len(exact_items)}")
        print(f"   🥈 Partial matches: {len(partial_items)}")
        print(f"   🥉 Semantic matches: {len(semantic_items)}")
        
        # Verify order
        if exact_items and results[0].get('match_type') == 'exact':
            print(f"   ✅ Exact matches are ranked first")
        elif not exact_items and partial_items and results[0].get('match_type') == 'partial':
            print(f"   ✅ Partial matches are ranked first (no exact matches)")
        elif not exact_items and not partial_items and semantic_items:
            print(f"   ✅ Semantic matches are ranked first (no exact/partial matches)")
        else:
            print(f"   ⚠️  Ranking might be incorrect")

def test_edge_cases():
    """Test edge cases"""
    print_header("🧪 Test 6: Edge Cases")
    
    test_cases = [
        ("", "Empty query"),
        ("xyz123", "Non-existent item"),
        ("ا", "Single character"),
        ("سمك مقلي مع رز قبولي وسلطة", "Very long query"),
    ]
    
    for query, description in test_cases:
        print(f"\n🔍 Testing: {description}")
        print(f"   Query: '{query}'")
        
        try:
            result = menu_search_service.search_menu(query)
            print(f"   ✅ No errors")
            print(f"   Found: {result.get('total_found', 0)} items")
        except Exception as e:
            print(f"   ❌ Error: {e}")

def main():
    """Run all tests"""
    print_header("🚀 Unified Menu Search Service Tests")
    
    try:
        # Run tests
        test_exact_match()
        test_partial_match()
        test_semantic_search()
        test_word_correction()
        test_priority_ranking()
        test_edge_cases()
        
        print_header("🎉 All Tests Completed!")
        
    except Exception as e:
        print(f"\n❌ Test failed with error: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    main()

