#!/usr/bin/env python3
"""
Test Text Cleaning for Search
اختبار تنظيف النص قبل البحث
"""

import os
import sys

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.utils.text_helpers import clean_query_for_search

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

def test_arabic_questions():
    """Test Arabic question words removal"""
    print_header("🧪 Test 1: Arabic Question Words")
    
    test_cases = [
        ("عندكم سمك؟", "سمك"),
        ("عندك هريس؟", "هريس"),
        ("فيه كبسة؟", "كبسة"),
        ("هل عندكم شوربة؟", "شوربة"),
        ("لو سمحت عندكم برجر؟", "برجر"),
        ("في سلطة فتوش؟", "سلطة فتوش"),
    ]
    
    passed = 0
    failed = 0
    
    for original, expected in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}' → '{cleaned}' (expected: '{expected}')")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_arabic_requests():
    """Test Arabic request words removal"""
    print_header("🧪 Test 2: Arabic Request Words")
    
    test_cases = [
        ("ابي سمك مقلي", "سمك مقلي"),
        ("أريد هريس دجاج", "هريس دجاج"),
        ("بدي كبسة عوان", "كبسة عوان"),
        ("عايز شاورما", "شاورما"),
        ("ممكن سلطة يونانية", "سلطة يونانية"),
        ("ابغى عصير برتقال", "عصير برتقال"),
    ]
    
    passed = 0
    failed = 0
    
    for original, expected in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}' → '{cleaned}' (expected: '{expected}')")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_english_questions():
    """Test English question words removal"""
    print_header("🧪 Test 3: English Question Words")
    
    test_cases = [
        ("do you have fish?", "fish"),
        ("have you got harees?", "harees"),
        ("is there kabsa?", "kabsa"),
        ("can i get a burger?", "burger"),
        ("could i have some salad?", "salad"),
        ("what is the soup?", "soup"),
    ]
    
    passed = 0
    failed = 0
    
    for original, expected in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}' → '{cleaned}' (expected: '{expected}')")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_english_requests():
    """Test English request words removal"""
    print_header("🧪 Test 4: English Request Words")
    
    test_cases = [
        ("i want fish", "fish"),
        ("i would like harees", "harees"),
        ("i'd like a burger please", "burger"),
        ("can i get chicken", "chicken"),
        ("please give me salad", "give me salad"),  # "give me" not in stopwords
    ]
    
    passed = 0
    failed = 0
    
    for original, expected in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}' → '{cleaned}' (expected: '{expected}')")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_complex_queries():
    """Test complex queries with multiple stopwords"""
    print_header("🧪 Test 5: Complex Queries")
    
    test_cases = [
        ("هل عندكم سمك مقلي مع رز قبولي؟", "سمك مقلي رز قبولي"),
        ("ابي هريس دجاج من فضلك", "هريس دجاج"),
        ("لو سمحت عندك كبسة عوان؟", "كبسة عوان"),
        ("do you have fried fish with rice?", "fried fish rice"),
        ("can i get chicken harees please?", "chicken harees"),
        ("i want kabsa awan", "kabsa awan"),
    ]
    
    passed = 0
    failed = 0
    
    for original, expected in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}'")
        print(f"   → '{cleaned}' (expected: '{expected}')")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_edge_cases():
    """Test edge cases"""
    print_header("🧪 Test 6: Edge Cases")
    
    test_cases = [
        ("سمك", "سمك", "Already clean"),
        ("fish", "fish", "Already clean"),
        ("عندكم", "", "Only stopwords"),
        ("do you have", "", "Only stopwords"),
        ("", "", "Empty string"),
        ("؟؟؟", "", "Only punctuation"),
    ]
    
    passed = 0
    failed = 0
    
    for original, expected, description in test_cases:
        cleaned = clean_query_for_search(original)
        
        if cleaned == expected:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        print(f"{icon} '{original}' → '{cleaned}' ({description})")
    
    print(f"\n📊 Summary: {passed}/{len(test_cases)} passed")
    return passed, failed

def test_real_world_examples():
    """Test real-world examples"""
    print_header("🧪 Test 7: Real-World Examples")
    
    test_cases = [
        "عندكم سمك؟",
        "هل فيه هريس دجاج؟",
        "ابي كبسة عوان لو سمحت",
        "ممكن شوربة عدس؟",
        "do you have fish?",
        "can i get chicken harees?",
        "i want kabsa please",
        "is there any salad?",
    ]
    
    print("\n📋 Real-world cleaning examples:")
    for query in test_cases:
        cleaned = clean_query_for_search(query)
        print(f"   '{query}'")
        print(f"   → '{cleaned}'")
        print()

def main():
    """Run all tests"""
    print_header("🚀 Text Cleaning Tests")
    
    # Run tests
    results = []
    results.append(test_arabic_questions())
    results.append(test_arabic_requests())
    results.append(test_english_questions())
    results.append(test_english_requests())
    results.append(test_complex_queries())
    results.append(test_edge_cases())
    
    # Real-world examples
    test_real_world_examples()
    
    # Overall summary
    total_passed = sum(r[0] for r in results)
    total_failed = sum(r[1] for r in results)
    total_tests = total_passed + total_failed
    
    print_header("🎯 Overall Summary")
    print(f"✅ Total Passed: {total_passed}/{total_tests}")
    print(f"❌ Total Failed: {total_failed}/{total_tests}")
    print(f"Success Rate: {total_passed/total_tests*100:.1f}%")
    
    if total_failed == 0:
        print("\n🎉 All tests passed!")
    else:
        print(f"\n⚠️  {total_failed} test(s) failed")

if __name__ == "__main__":
    main()

