"""
Test Structured Outputs Implementation
This script tests the new structured outputs system
"""
import os
import sys
import asyncio
from pathlib import Path

# Add parent directory to path
sys.path.append(str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from app.services.openai_service_structured import OpenAIServiceStructured
from app.models.ai_responses import AddItemResponse, AskInformationResponse, SmallTalkResponse

load_dotenv()


async def test_single_item():
    """Test adding a single item (should use AddItemResponse)"""
    print("\n" + "="*80)
    print("TEST 1: Single Item (حمص)")
    print("="*80)
    
    service = OpenAIServiceStructured()
    
    system_prompt = """You are a restaurant assistant.

When customer requests an item:
1. If exactly 1 match → Add it using AddItemResponse
2. If multiple matches → Ask using AskInformationResponse

Menu:
- حمص (Hummus) - 1.2 OMR - Code: APP_HUMMUS
"""
    
    user_message = "Customer Message: حمص"
    
    try:
        response = await service.get_structured_response(
            system_prompt=system_prompt,
            user_message=user_message,
            response_model=AddItemResponse
        )
        
        print(f"\n✅ Response Type: {type(response).__name__}")
        print(f"✅ Intent: {response.intent}")
        print(f"✅ Items: {len(response.items)}")
        print(f"✅ Item Name: {response.items[0].name}")
        print(f"✅ Quantity: {response.items[0].quantity}")
        print(f"✅ Price: {response.items[0].price}")
        print(f"✅ Subtotal: {response.subtotal}")
        print(f"✅ Reply: {response.reply[:100]}...")
        
        assert response.intent == "New Order - Items"
        assert len(response.items) == 1
        assert response.items[0].quantity >= 1
        assert response.subtotal > 0
        
        print("\n✅ TEST 1 PASSED!")
        return True
        
    except Exception as e:
        print(f"\n❌ TEST 1 FAILED: {e}")
        return False


async def test_multiple_items():
    """Test multiple items (should use AskInformationResponse)"""
    print("\n" + "="*80)
    print("TEST 2: Multiple Items (سمك)")
    print("="*80)
    
    service = OpenAIServiceStructured()
    
    system_prompt = """You are a restaurant assistant.

When customer requests an item:
1. If exactly 1 match → Add it using AddItemResponse
2. If multiple matches → Ask using AskInformationResponse
   - Reply MUST contain numbered list (1. ... 2. ...)
   - Format: "لدينا نوعان:\n\n1. [name] - [price]\n2. [name] - [price]\n\nأي نوع تفضل؟"

Menu:
- مصلونة سمك تونة مع رز أبيض - 3.5 OMR - Code: MAIN_MASLOONA_TUNA
- سمك مقلي مع رز قبولي - 3.5 OMR - Code: MAIN_FRIED_FISH
"""
    
    user_message = "Customer Message: سمك"
    
    try:
        response = await service.get_structured_response(
            system_prompt=system_prompt,
            user_message=user_message,
            response_model=AskInformationResponse
        )
        
        print(f"\n✅ Response Type: {type(response).__name__}")
        print(f"✅ Intent: {response.intent}")
        print(f"✅ Options: {len(response.options)}")
        print(f"✅ Reply: {response.reply[:200]}...")
        
        assert response.intent == "Ask Information", f"Expected 'Ask Information', got '{response.intent}'"
        assert len(response.options) >= 2, f"Expected at least 2 options, got {len(response.options)}"
        assert '1.' in response.reply or '1-' in response.reply or '١' in response.reply, f"Reply doesn't contain numbered list: {response.reply}"
        assert '2.' in response.reply or '2-' in response.reply or '٢' in response.reply, f"Reply doesn't contain second option: {response.reply}"
        
        print("\n✅ TEST 2 PASSED!")
        return True
        
    except Exception as e:
        print(f"\n❌ TEST 2 FAILED: {e}")
        return False


async def test_small_talk():
    """Test small talk (should use SmallTalkResponse)"""
    print("\n" + "="*80)
    print("TEST 3: Small Talk (السلام عليكم)")
    print("="*80)
    
    service = OpenAIServiceStructured()
    
    system_prompt = """You are a restaurant assistant.

When customer greets you, respond with SmallTalkResponse."""
    
    user_message = "Customer Message: السلام عليكم"
    
    try:
        response = await service.get_structured_response(
            system_prompt=system_prompt,
            user_message=user_message,
            response_model=SmallTalkResponse
        )
        
        print(f"\n✅ Response Type: {type(response).__name__}")
        print(f"✅ Intent: {response.intent}")
        print(f"✅ Reply: {response.reply}")
        
        assert response.intent == "Small Talk"
        assert len(response.reply) > 0
        
        print("\n✅ TEST 3 PASSED!")
        return True
        
    except Exception as e:
        print(f"\n❌ TEST 3 FAILED: {e}")
        return False


async def test_context_awareness():
    """Test context awareness with conversation history"""
    print("\n" + "="*80)
    print("TEST 4: Context Awareness (selecting from options)")
    print("="*80)
    
    service = OpenAIServiceStructured()
    
    system_prompt = """You are a restaurant assistant.

CRITICAL: Review conversation history!
- If customer says "1", "2", "الاول", "الثاني" → they're selecting from your last message
- Add the selected item using AddItemResponse

Menu:
- مصلونة سمك تونة مع رز أبيض - 3.5 OMR - Code: MAIN_MASLOONA_TUNA
- سمك مقلي مع رز قبولي - 3.5 OMR - Code: MAIN_FRIED_FISH
"""
    
    conversation_history = [
        {
            "role": "user",
            "content": "Customer Message: سمك"
        },
        {
            "role": "assistant",
            "content": "لدينا نوعان من السمك:\n\n1. مصلونة سمك تونة مع رز أبيض - 3.500 ر.ع\n2. سمك مقلي مع رز قبولي - 3.500 ر.ع\n\nأي نوع تفضل؟"
        }
    ]
    
    user_message = "Customer Message: 2"
    
    try:
        response = await service.get_structured_response(
            system_prompt=system_prompt,
            user_message=user_message,
            response_model=AddItemResponse,
            conversation_history=conversation_history
        )
        
        print(f"\n✅ Response Type: {type(response).__name__}")
        print(f"✅ Intent: {response.intent}")
        print(f"✅ Item Name: {response.items[0].name}")
        print(f"✅ Item Code: {response.items[0].code}")
        print(f"✅ Reply: {response.reply[:100]}...")
        
        assert response.intent == "New Order - Items"
        assert "سمك مقلي" in response.items[0].name or response.items[0].code == "MAIN_FRIED_FISH"
        
        print("\n✅ TEST 4 PASSED!")
        return True
        
    except Exception as e:
        print(f"\n❌ TEST 4 FAILED: {e}")
        import traceback
        traceback.print_exc()
        return False


async def main():
    """Run all tests"""
    print("\n🧪 Testing Structured Outputs Implementation")
    print("="*80)
    
    results = []
    
    # Run tests
    results.append(await test_single_item())
    results.append(await test_multiple_items())
    results.append(await test_small_talk())
    results.append(await test_context_awareness())
    
    # Summary
    print("\n" + "="*80)
    print("📊 TEST SUMMARY")
    print("="*80)
    
    passed = sum(results)
    total = len(results)
    
    print(f"\nPassed: {passed}/{total}")
    print(f"Failed: {total - passed}/{total}")
    
    if passed == total:
        print("\n✅ ALL TESTS PASSED!")
        print("\n💡 Next step: Integrate with webhook.py")
    else:
        print("\n❌ SOME TESTS FAILED!")
        print("\n💡 Please review the errors above")
    
    return passed == total


if __name__ == '__main__':
    success = asyncio.run(main())
    sys.exit(0 if success else 1)

