#!/usr/bin/env python3
"""
Test SHOW_MENU intent exclusions
اختبار استثناءات نية عرض المنيو
"""

import os
import sys

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.intents.rule_based_classifier import RuleBasedClassifier
from app.intents.intent_types import IntentType

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*70)
    print(f"  {text}")
    print("="*70)

def print_result(message: str, result: dict):
    """Print test result"""
    intent = result.get('intent')
    confidence = result.get('confidence', 0)
    method = result.get('method', 'unknown')
    
    icon = "✅" if intent != IntentType.SHOW_MENU else "❌"
    
    print(f"\n{icon} '{message}'")
    print(f"   Intent: {intent}")
    print(f"   Confidence: {confidence:.2f}")
    print(f"   Method: {method}")

def main():
    print_header("🧪 Testing SHOW_MENU Exclusions")
    
    classifier = RuleBasedClassifier()
    
    # Test cases that SHOULD trigger SHOW_MENU
    print_header("✅ Should be SHOW_MENU (no food names)")
    
    should_be_menu = [
        "المنيو",
        "عندكم ايش؟",
        "شو القائمة؟",
        "أريد أشوف القائمة",
        "menu please",
        "show me the menu",
        "what do you have?",
    ]
    
    for message in should_be_menu:
        result = classifier.classify(message, {})
        if result:
            intent, confidence = result
            expected = intent == IntentType.SHOW_MENU
            icon = "✅" if expected else "❌"
            print(f"{icon} '{message}' → {intent} ({confidence:.2f})")
        else:
            print(f"⚠️  '{message}' → No match")
    
    # Test cases that SHOULD NOT trigger SHOW_MENU (food names)
    print_header("❌ Should NOT be SHOW_MENU (contains food names)")
    
    should_not_be_menu = [
        # Arabic food names
        "سمك",
        "دجاج",
        "هريس",
        "كبسة",
        "برجر",
        "شاورما",
        "فتوش",
        "حمص",
        "عصير",
        "لبن",
        "أريد هريس دجاج",
        "عندكم سمك؟",
        "بغيت برجر",
        # English food names
        "fish",
        "chicken",
        "harees",
        "kabsa",
        "burger",
        "shawarma",
        "fattoush",
        "hummus",
        "juice",
        "I want chicken",
        "do you have fish?",
        "burger please",
    ]
    
    for message in should_not_be_menu:
        result = classifier.classify(message, {})
        if result:
            intent, confidence = result
            expected = intent != IntentType.SHOW_MENU
            icon = "✅" if expected else "❌"
            print(f"{icon} '{message}' → {intent} ({confidence:.2f}) - Expected: NOT SHOW_MENU")
        else:
            print(f"✅ '{message}' → No match (Good! Not SHOW_MENU)")
    
    # Summary
    print_header("📊 Summary")
    print("✅ = Test passed")
    print("❌ = Test failed")
    print("\nExpected behavior:")
    print("  - 'المنيو' → SHOW_MENU")
    print("  - 'سمك' → ORDER_FOOD (not SHOW_MENU)")
    print("  - 'عندكم ايش؟' → SHOW_MENU")
    print("  - 'عندكم سمك؟' → ORDER_FOOD (not SHOW_MENU)")

if __name__ == "__main__":
    main()

