#!/usr/bin/env python3
"""
Test Search Stability
اختبار استقرار نتائج البحث
"""

import os
import sys
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.services.menu_search_service import menu_search_service

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

def print_results(result: dict):
    """Print search results"""
    print(f"\nFound {result['total_found']} results:")
    print(f"Methods: {', '.join(result['search_methods'])}")
    print(f"Exact: {result['exact_count']}, Partial: {result['partial_count']}, Semantic: {result['semantic_count']}")
    print("\nResults:")
    for i, item in enumerate(result['results'], 1):
        name_ar = item['name_ar']
        match_type = item['match_type']
        match_score = item['match_score']
        print(f"  {i}. {name_ar} ({match_type}, score={match_score:.2f})")

def test_stability():
    """Test search stability"""
    print_header("🧪 Test Search Stability")
    
    queries = [
        "هريس",
        "عندكم هريس؟",
        "هريس",  # Same as first
        "عندكم هريس",  # Similar to second
        "هريس",  # Same as first again
    ]
    
    results = []
    
    for i, query in enumerate(queries, 1):
        print_header(f"Test {i}: '{query}'")
        result = menu_search_service.search_menu(query, limit=5)
        print_results(result)
        results.append(result)
    
    # Check consistency
    print_header("🔍 Consistency Check")
    
    # Compare Test 1, 3, 5 (all "هريس")
    test1_items = [item['code'] for item in results[0]['results']]
    test3_items = [item['code'] for item in results[2]['results']]
    test5_items = [item['code'] for item in results[4]['results']]
    
    if test1_items == test3_items == test5_items:
        print("✅ Tests 1, 3, 5 are CONSISTENT (same query → same results)")
    else:
        print("❌ Tests 1, 3, 5 are INCONSISTENT (same query → different results)")
        print(f"   Test 1: {test1_items}")
        print(f"   Test 3: {test3_items}")
        print(f"   Test 5: {test5_items}")
    
    # Compare Test 2, 4 (both "عندكم هريس")
    test2_items = [item['code'] for item in results[1]['results']]
    test4_items = [item['code'] for item in results[3]['results']]
    
    if test2_items == test4_items:
        print("✅ Tests 2, 4 are CONSISTENT (similar query → same results)")
    else:
        print("❌ Tests 2, 4 are INCONSISTENT (similar query → different results)")
        print(f"   Test 2: {test2_items}")
        print(f"   Test 4: {test4_items}")

if __name__ == "__main__":
    test_stability()

