#!/usr/bin/env python3
"""
Test Product Inquiry Detection
اختبار اكتشاف الاستفسار عن الأصناف
"""

import os
import sys

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.utils.order_intent_classifier import order_intent_classifier

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*70)
    print(f"  {text}")
    print("="*70)

def test_product_inquiry():
    """Test detect_product_inquiry function"""
    
    print_header("🧪 Testing Product Inquiry Detection")
    
    # Test cases: (message, expected_has_product)
    test_cases = [
        # Should detect products
        ("سمك", True),
        ("دجاج", True),
        ("هريس", True),
        ("كبسة", True),
        ("برجر", True),
        ("عندكم سمك؟", True),
        ("عندكم دجاج؟", True),
        ("في هريس؟", True),
        ("موجود برجر؟", True),
        ("أريد هريس دجاج", True),
        ("بغيت سمك مقلي", True),
        ("عندكم كبسة عوان؟", True),
        ("fish", True),
        ("chicken", True),
        ("harees", True),
        ("kabsa", True),
        ("burger", True),
        ("do you have fish?", True),
        ("I want chicken", True),
        ("burger please", True),
        ("كم سعر السمك؟", True),
        ("بكم الدجاج؟", True),
        ("سمك مقلي مع رز قبولي", True),
        ("هريس دجاج", True),
        ("مصلونة سمك تونة", True),
        ("عرسي جعلان", True),
        ("قشاع دجاج", True),
        ("روبيان مقلي", True),
        ("حمص", True),
        ("فتوش", True),
        ("تبولة", True),
        ("عصير برتقال", True),
        ("قهوة عربية", True),
        ("شاي أحمر", True),
        
        # Should NOT detect products
        ("المنيو", False),
        ("عندكم ايش؟", False),
        ("menu please", False),
        ("what do you have?", False),
        ("كم السعر؟", False),
        ("مرحبا", False),
        ("شكراً", False),
        ("hello", False),
        ("thanks", False),
        ("أريد أشوف القائمة", False),
        ("وريني المنيو", False),
    ]
    
    passed = 0
    failed = 0
    
    print("\n✅ Should detect products:")
    print("-" * 70)
    
    for message, expected_has_product in test_cases:
        if expected_has_product:
            result = order_intent_classifier.detect_product_inquiry(message)
            
            if result['has_product'] == expected_has_product:
                icon = "✅"
                passed += 1
            else:
                icon = "❌"
                failed += 1
            
            products_str = ", ".join(result['products_found'][:3])
            if len(result['products_found']) > 3:
                products_str += f" (+{len(result['products_found']) - 3} more)"
            
            print(f"{icon} '{message}'")
            print(f"   Has product: {result['has_product']}")
            print(f"   Products: {products_str}")
            print(f"   Confidence: {result['confidence']}")
            print()
    
    print("\n❌ Should NOT detect products:")
    print("-" * 70)
    
    for message, expected_has_product in test_cases:
        if not expected_has_product:
            result = order_intent_classifier.detect_product_inquiry(message)
            
            if result['has_product'] == expected_has_product:
                icon = "✅"
                passed += 1
            else:
                icon = "❌"
                failed += 1
            
            print(f"{icon} '{message}'")
            print(f"   Has product: {result['has_product']} (expected: False)")
            if result['products_found']:
                print(f"   False positives: {result['products_found'][:3]}")
            print()
    
    # Summary
    print_header("📊 Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"❌ Failed: {failed}/{len(test_cases)}")
    print(f"Success Rate: {passed/len(test_cases)*100:.1f}%")
    
    if failed == 0:
        print("\n🎉 All tests passed!")
    else:
        print(f"\n⚠️  {failed} test(s) failed")

if __name__ == "__main__":
    test_product_inquiry()

