#!/usr/bin/env python3
"""
Test Menu Sync System
Tests all components of the menu sync flow
"""
import sys
import os
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

# Load environment variables
from dotenv import load_dotenv
load_dotenv()

from app.database.supabase_client import get_supabase_client
from app.cache.menu_cache import get_menu_cache
from app.services.menu_sync_service import get_menu_sync_service
from app.embeddings.text_cleaning import clean_text, normalize_arabic, prepare_text_for_embedding
from app.utils.logger import logger


def print_section(title: str):
    """Print section header"""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80)


def test_supabase_client():
    """Test Supabase client"""
    print_section("🗄️  TEST 1: Supabase Client")
    
    try:
        client = get_supabase_client()
        
        # Health check
        print("\n1. Health Check...")
        healthy = client.health_check()
        print(f"   {'✅' if healthy else '❌'} Supabase connection: {'Healthy' if healthy else 'Failed'}")
        
        # Get menu items
        print("\n2. Fetching menu items...")
        items = client.get_all_menu_items(active_only=True)
        print(f"   ✅ Fetched {len(items)} items")
        
        # Show sample item
        if items:
            print("\n3. Sample item:")
            sample = items[0]
            print(f"   Code: {sample.get('code')}")
            print(f"   Name AR: {sample.get('name_ar')}")
            print(f"   Name EN: {sample.get('name_en')}")
            print(f"   Description: {sample.get('description', 'N/A')[:50]}...")
            print(f"   Tags: {sample.get('tags', [])}")
            print(f"   Category: {sample.get('category')}")
            print(f"   Price: {sample.get('price_omr')} OMR")
        
        # Get by category
        print("\n4. Testing category filter...")
        omani_items = client.get_menu_items_by_category('أطباق عمانية')
        print(f"   ✅ Found {len(omani_items)} Omani dishes")
        
        # Search
        print("\n5. Testing search...")
        results = client.search_menu_items('هريس', limit=5)
        print(f"   ✅ Found {len(results)} items matching 'هريس'")
        
        print("\n✅ Supabase client tests passed!")
        return True
        
    except Exception as e:
        print(f"\n❌ Supabase client test failed: {str(e)}")
        return False


def test_text_cleaning():
    """Test text cleaning utilities"""
    print_section("🧹 TEST 2: Text Cleaning")
    
    try:
        # Test clean_text
        print("\n1. Testing clean_text()...")
        dirty_text = "  هَرِيس  دَجَاج  <b>مع</b>  رز   "
        clean = clean_text(dirty_text)
        print(f"   Input:  '{dirty_text}'")
        print(f"   Output: '{clean}'")
        print(f"   ✅ Cleaned successfully")
        
        # Test normalize_arabic
        print("\n2. Testing normalize_arabic()...")
        text_with_variants = "أكلة إماراتية آسيوية"
        normalized = normalize_arabic(text_with_variants)
        print(f"   Input:  '{text_with_variants}'")
        print(f"   Output: '{normalized}'")
        print(f"   ✅ Normalized successfully")
        
        # Test prepare_text_for_embedding
        print("\n3. Testing prepare_text_for_embedding()...")
        sample_item = {
            'code': 'TEST',
            'name_ar': 'هريس دجاج',
            'name_en': 'Harees Chicken',
            'description': 'طبق عماني تقليدي',
            'tags': ['هريس', 'دجاج', 'عماني'],
            'category': 'أطباق عمانية'
        }
        embedding_text = prepare_text_for_embedding(sample_item)
        print(f"   Output: '{embedding_text[:100]}...'")
        print(f"   Length: {len(embedding_text)} characters")
        print(f"   ✅ Prepared successfully")
        
        print("\n✅ Text cleaning tests passed!")
        return True
        
    except Exception as e:
        print(f"\n❌ Text cleaning test failed: {str(e)}")
        return False


def test_menu_cache():
    """Test menu cache"""
    print_section("💾 TEST 3: Menu Cache")
    
    try:
        cache = get_menu_cache()
        
        # Get stats
        print("\n1. Cache statistics...")
        stats = cache.get_stats()
        print(f"   Loaded: {stats['loaded']}")
        print(f"   Items: {stats['item_count']}")
        print(f"   Categories: {stats['category_count']}")
        print(f"   Last Update: {stats['last_update']}")
        print(f"   Stale: {stats['is_stale']}")
        print(f"   File Exists: {stats['cache_file_exists']}")
        
        # Load from file
        print("\n2. Loading from file...")
        loaded = cache.load_from_file()
        print(f"   {'✅' if loaded else '⚠️ '} Load result: {loaded}")
        
        if loaded:
            # Get all items
            print("\n3. Getting all items...")
            items = cache.get_all_items()
            print(f"   ✅ Retrieved {len(items)} items")
            
            # Get by code
            print("\n4. Testing get by code...")
            item = cache.get_item_by_code('HAREES_CHICKEN')
            if item:
                print(f"   ✅ Found: {item['name_ar']}")
            else:
                print(f"   ⚠️  Item not found")
            
            # Get categories
            print("\n5. Getting categories...")
            categories = cache.get_categories()
            print(f"   ✅ Found {len(categories)} categories:")
            for cat in categories:
                count = len(cache.get_items_by_category(cat))
                print(f"      - {cat}: {count} items")
            
            # Search
            print("\n6. Testing search...")
            results = cache.search_items('هريس')
            print(f"   ✅ Found {len(results)} items matching 'هريس'")
        
        print("\n✅ Menu cache tests passed!")
        return True
        
    except Exception as e:
        print(f"\n❌ Menu cache test failed: {str(e)}")
        return False


def test_menu_sync_service():
    """Test menu sync service"""
    print_section("🔄 TEST 4: Menu Sync Service")
    
    try:
        sync_service = get_menu_sync_service()
        
        # Get status
        print("\n1. Getting sync status...")
        status = sync_service.get_sync_status()
        
        print(f"\n   Supabase:")
        print(f"      Healthy: {status['supabase']['healthy']}")
        print(f"      Items: {status['supabase']['item_count']}")
        
        print(f"\n   Cache:")
        print(f"      Loaded: {status['cache']['loaded']}")
        print(f"      Items: {status['cache']['item_count']}")
        print(f"      Stale: {status['cache']['is_stale']}")
        
        print(f"\n   FAISS:")
        print(f"      Exists: {status['faiss']['exists']}")
        print(f"      Size: {status['faiss']['size_bytes']} bytes")
        
        print(f"\n   Sync Needed: {status['sync_needed']}")
        
        # Validate sync
        print("\n2. Validating sync...")
        validation = sync_service.validate_sync()
        print(f"   Valid: {validation['valid']}")
        
        if validation['issues']:
            print(f"   Issues:")
            for issue in validation['issues']:
                print(f"      - {issue}")
        
        if validation['warnings']:
            print(f"   Warnings:")
            for warning in validation['warnings']:
                print(f"      - {warning}")
        
        print("\n✅ Menu sync service tests passed!")
        return True
        
    except Exception as e:
        print(f"\n❌ Menu sync service test failed: {str(e)}")
        return False


def main():
    """Run all tests"""
    print("="*80)
    print("🧪 MENU SYNC SYSTEM TEST SUITE")
    print("="*80)
    
    results = {
        'Supabase Client': test_supabase_client(),
        'Text Cleaning': test_text_cleaning(),
        'Menu Cache': test_menu_cache(),
        'Menu Sync Service': test_menu_sync_service()
    }
    
    # Summary
    print_section("📊 TEST SUMMARY")
    
    passed = sum(1 for result in results.values() if result)
    total = len(results)
    
    print()
    for test_name, result in results.items():
        status = "✅ PASSED" if result else "❌ FAILED"
        print(f"   {status}: {test_name}")
    
    print(f"\n   Total: {passed}/{total} tests passed")
    
    if passed == total:
        print("\n🎉 All tests passed!")
        return 0
    else:
        print(f"\n⚠️  {total - passed} test(s) failed")
        return 1


if __name__ == '__main__':
    exit_code = main()
    sys.exit(exit_code)

