#!/usr/bin/env python3
"""
Test Layered Architecture - اختبار البنية الطبقية
اختبار شامل لجميع المكونات الجديدة

الاستخدام:
    python3 scripts/test_layered_architecture.py
"""

import asyncio
import sys
import os
from pathlib import Path

# Add project root to path
project_root = Path(__file__).parent.parent
sys.path.insert(0, str(project_root))

from app.services.input_processor import input_processor
from app.services.unified_menu_search import unified_menu_search
from app.services.response_formatter import response_formatter, ResponseType
from app.services.analytics_service import analytics_service
from app.utils.logger import app_logger as logger


class LayeredArchitectureTest:
    """اختبار البنية الطبقية"""
    
    def __init__(self):
        self.passed = 0
        self.failed = 0
        self.total = 0
    
    def print_header(self, title: str):
        """طباعة عنوان"""
        print("\n" + "=" * 70)
        print(f"  {title}")
        print("=" * 70)
    
    def print_test(self, test_name: str, passed: bool, details: str = ""):
        """طباعة نتيجة اختبار"""
        self.total += 1
        if passed:
            self.passed += 1
            status = "✅ PASS"
        else:
            self.failed += 1
            status = "❌ FAIL"
        
        print(f"{status} | {test_name}")
        if details:
            print(f"       {details}")
    
    async def test_input_processor(self):
        """اختبار معالج الإدخال"""
        self.print_header("🟦 LAYER 1: INPUT PROCESSOR")
        
        # Test 1: Basic processing
        try:
            result = await input_processor.process(
                message_text="عندكم هرس دجاج؟",
                chat_id="96812345678@s.whatsapp.net",
                phone_number="96812345678",
                message_id="test_msg_1"
            )
            
            self.print_test(
                "Basic text processing",
                result.cleaned_text is not None,
                f"Original: '{result.original_text}' → Cleaned: '{result.cleaned_text}'"
            )
            
            self.print_test(
                "Language detection",
                result.language == "ar",
                f"Language: {result.language}, Dialect: {result.dialect}"
            )
            
            self.print_test(
                "Word correction",
                result.corrections_made >= 0,
                f"Corrections: {result.corrections_made}"
            )
            
            self.print_test(
                "Processing time",
                result.processing_time_ms < 100,
                f"Time: {result.processing_time_ms:.1f}ms"
            )
            
        except Exception as e:
            self.print_test("Input processor", False, f"Error: {str(e)}")
        
        # Test 2: English text
        try:
            result = await input_processor.process(
                message_text="Do you have chicken harees?",
                chat_id="96812345678@s.whatsapp.net",
                phone_number="96812345678",
                message_id="test_msg_2"
            )
            
            self.print_test(
                "English text processing",
                result.language == "en",
                f"Language: {result.language}"
            )
            
        except Exception as e:
            self.print_test("English processing", False, f"Error: {str(e)}")
    
    async def test_unified_menu_search(self):
        """اختبار البحث الموحد"""
        self.print_header("🟩 LAYER 3: UNIFIED MENU SEARCH")
        
        # Test 1: Basic search
        try:
            result = await unified_menu_search.search(
                query="هريس دجاج",
                top_k=5
            )
            
            self.print_test(
                "Basic menu search",
                result.total_results >= 0,
                f"Found: {result.total_results} items in {result.search_time_ms:.1f}ms"
            )
            
            self.print_test(
                "Search method detection",
                result.method is not None,
                f"Method: {result.method.value}"
            )
            
            self.print_test(
                "Query correction",
                result.query is not None,
                f"Query: '{result.query}' → '{result.corrected_query or result.query}'"
            )
            
        except Exception as e:
            self.print_test("Menu search", False, f"Error: {str(e)}")
        
        # Test 2: Search with filters
        try:
            result = await unified_menu_search.search(
                query="دجاج",
                top_k=3,
                max_price=5.0
            )
            
            self.print_test(
                "Search with price filter",
                result.total_results >= 0,
                f"Found: {result.total_results} items (max price: 5.0 OMR)"
            )
            
        except Exception as e:
            self.print_test("Filtered search", False, f"Error: {str(e)}")
    
    async def test_response_formatter(self):
        """اختبار منسق الردود"""
        self.print_header("🟥 LAYER 4: RESPONSE FORMATTER")
        
        # Test 1: Menu list formatting
        try:
            test_items = [
                {
                    'name_ar': 'هريس دجاج',
                    'name_en': 'Chicken Harees',
                    'price_omr': 2.5,
                    'description_ar': 'هريس دجاج لذيذ'
                },
                {
                    'name_ar': 'هريس لحم',
                    'name_en': 'Meat Harees',
                    'price_omr': 3.0,
                    'description_ar': 'هريس لحم طازج'
                }
            ]
            
            response = response_formatter.format_menu_list(
                items=test_items,
                language="ar",
                show_prices=True
            )
            
            self.print_test(
                "Menu list formatting (Arabic)",
                response.text is not None and len(response.text) > 0,
                f"Length: {len(response.text)} chars"
            )
            
            self.print_test(
                "Response type",
                response.response_type == ResponseType.MENU_LIST,
                f"Type: {response.response_type.value}"
            )
            
            # English version
            response_en = response_formatter.format_menu_list(
                items=test_items,
                language="en",
                show_prices=True
            )
            
            self.print_test(
                "Menu list formatting (English)",
                response_en.text is not None,
                f"Length: {len(response_en.text)} chars"
            )
            
        except Exception as e:
            self.print_test("Menu list formatting", False, f"Error: {str(e)}")
        
        # Test 2: Cart summary formatting
        try:
            cart_items = [
                {
                    'name_ar': 'هريس دجاج',
                    'name_en': 'Chicken Harees',
                    'quantity': 2,
                    'price': 2.5
                },
                {
                    'name_ar': 'عصير برتقال',
                    'name_en': 'Orange Juice',
                    'quantity': 1,
                    'price': 1.0
                }
            ]
            
            response = response_formatter.format_cart_summary(
                items=cart_items,
                subtotal=6.0,
                language="ar"
            )
            
            self.print_test(
                "Cart summary formatting",
                response.text is not None and "6.0" in response.text,
                f"Type: {response.response_type.value}"
            )
            
        except Exception as e:
            self.print_test("Cart formatting", False, f"Error: {str(e)}")
        
        # Test 3: Order confirmation formatting
        try:
            response = response_formatter.format_order_confirmation(
                order_number="ORD-12345",
                items=cart_items,
                total=6.0,
                address="مسقط، الخوير",
                payment_method="نقداً",
                language="ar"
            )
            
            self.print_test(
                "Order confirmation formatting",
                response.text is not None and "ORD-12345" in response.text,
                f"Type: {response.response_type.value}"
            )
            
        except Exception as e:
            self.print_test("Order confirmation", False, f"Error: {str(e)}")
    
    async def test_analytics_service(self):
        """اختبار خدمة التحليلات"""
        self.print_header("🟥 LAYER 4: ANALYTICS SERVICE")
        
        # Test 1: Track intent
        try:
            await analytics_service.track_intent(
                intent_type="ORDER_FOOD",
                response_time_ms=450.0,
                classification_method="rule_based",
                success=True,
                cost_usd=0.0
            )
            
            self.print_test(
                "Track intent",
                True,
                "Intent tracked successfully"
            )
            
        except Exception as e:
            self.print_test("Track intent", False, f"Error: {str(e)}")
        
        # Test 2: Track user activity
        try:
            await analytics_service.track_user_activity(
                user_id="test_user_1",
                intent_type="ORDER_FOOD"
            )
            
            self.print_test(
                "Track user activity",
                True,
                "User activity tracked"
            )
            
        except Exception as e:
            self.print_test("Track user", False, f"Error: {str(e)}")
        
        # Test 3: Get performance metrics
        try:
            metrics = await analytics_service.get_performance_metrics()
            
            self.print_test(
                "Get performance metrics",
                metrics.total_messages >= 0,
                f"Total messages: {metrics.total_messages}"
            )
            
        except Exception as e:
            self.print_test("Performance metrics", False, f"Error: {str(e)}")
        
        # Test 4: Generate daily report
        try:
            report = await analytics_service.generate_daily_report()
            
            self.print_test(
                "Generate daily report",
                'summary' in report,
                f"Report keys: {list(report.keys())}"
            )
            
        except Exception as e:
            self.print_test("Daily report", False, f"Error: {str(e)}")
    
    def print_summary(self):
        """طباعة الملخص"""
        self.print_header("📊 TEST SUMMARY")
        
        print(f"\nTotal Tests:  {self.total}")
        print(f"✅ Passed:    {self.passed}")
        print(f"❌ Failed:    {self.failed}")
        
        if self.failed == 0:
            print("\n🎉 All tests passed!")
        else:
            print(f"\n⚠️  {self.failed} test(s) failed")
        
        success_rate = (self.passed / self.total * 100) if self.total > 0 else 0
        print(f"\nSuccess Rate: {success_rate:.1f}%")
        
        print("\n" + "=" * 70)


async def main():
    """الدالة الرئيسية"""
    print("\n🧪 Testing Layered Architecture Components")
    print("=" * 70)
    
    tester = LayeredArchitectureTest()
    
    # Run all tests
    await tester.test_input_processor()
    await tester.test_unified_menu_search()
    await tester.test_response_formatter()
    await tester.test_analytics_service()
    
    # Print summary
    tester.print_summary()
    
    # Exit with appropriate code
    sys.exit(0 if tester.failed == 0 else 1)


if __name__ == "__main__":
    asyncio.run(main())

