#!/usr/bin/env python3
"""
Intent System Test Script
سكريبت اختبار نظام النوايا
"""

import sys
import os
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.intents import initialize_intent_system, unified_classifier
from app.intents.intent_types import IntentType
import time


def print_header(text: str):
    """طباعة عنوان"""
    print("\n" + "=" * 60)
    print(f"  {text}")
    print("=" * 60)


def print_result(message: str, result):
    """طباعة نتيجة الاختبار"""
    print(f"\n📝 Message: {message}")
    
    if result:
        intent_type, confidence, reason = result
        print(f"✅ Intent: {intent_type.value}")
        print(f"   Confidence: {confidence:.2f}")
        print(f"   Reason: {reason}")
    else:
        print("❌ No match found")


def test_greetings():
    """اختبار التحيات"""
    print_header("Testing Greetings")
    
    test_cases = [
        "مرحبا",
        "السلام عليكم",
        "صباح الخير",
        "Hello",
        "Hi there",
        "Good morning"
    ]
    
    for message in test_cases:
        result = unified_classifier.classify(message, {})
        print_result(message, result)


def test_confirmations():
    """اختبار التأكيدات"""
    print_header("Testing Confirmations")
    
    test_cases = [
        ("نعم", {'bot_asked_question': True}),
        ("أيوه", {'bot_asked_question': True}),
        ("لا", {'bot_asked_question': True}),
        ("Yes", {'bot_asked_question': True}),
        ("No", {'bot_asked_question': True})
    ]
    
    for message, context in test_cases:
        result = unified_classifier.classify(message, context)
        print_result(message, result)


def test_menu_requests():
    """اختبار طلبات المنيو"""
    print_header("Testing Menu Requests")
    
    test_cases = [
        "أريد المنيو",
        "عندكم ايش؟",
        "شو القائمة؟",
        "Show me the menu",
        "What do you have?"
    ]
    
    for message in test_cases:
        result = unified_classifier.classify(message, {})
        print_result(message, result)


def test_price_questions():
    """اختبار أسئلة الأسعار"""
    print_header("Testing Price Questions")
    
    test_cases = [
        "كم سعر الهريس؟",
        "بكم الكبسة؟",
        "How much is the hummus?",
        "Price of chicken?"
    ]
    
    for message in test_cases:
        result = unified_classifier.classify(message, {})
        print_result(message, result)


def test_order_food():
    """اختبار طلب الطعام"""
    print_header("Testing Food Orders")
    
    test_cases = [
        "أريد 2 هريس لحم",
        "بغيت كبسة دجاج",
        "عايز سلطة فتوش",
        "I want 2 chicken shawarma",
        "Order hummus please"
    ]
    
    for message in test_cases:
        result = unified_classifier.classify(message, {})
        print_result(message, result)


def test_cart_operations():
    """اختبار عمليات السلة"""
    print_header("Testing Cart Operations")
    
    test_cases = [
        ("وش طلبي؟", {}),
        ("عرض السلة", {}),
        ("أكمل الطلب", {'items': [{'name': 'كبسة'}]}),
        ("Show my cart", {}),
        ("Complete order", {'items': [{'name': 'hummus'}]})
    ]
    
    for message, context in test_cases:
        result = unified_classifier.classify(message, context)
        print_result(message, result)


def test_special_patterns():
    """اختبار الأنماط الخاصة"""
    print_header("Testing Special Patterns")
    
    test_cases = [
        ("1", {'bot_presented_options': True}),
        ("2", {'bot_presented_options': True}),
        ("ORD-123456", {}),
        ("تتبع الطلب ORD-789012", {})
    ]
    
    for message, context in test_cases:
        result = unified_classifier.classify(message, context)
        print_result(message, result)


def test_context_awareness():
    """اختبار الوعي بالسياق"""
    print_header("Testing Context Awareness")
    
    # Test "add more items" context
    test_cases = [
        ("نعم", {'waiting_for_more_items': True}),
        ("لا", {'waiting_for_more_items': True}),
        ("خلاص", {'waiting_for_more_items': True})
    ]
    
    for message, context in test_cases:
        result = unified_classifier.classify(message, context)
        print_result(message, result)


def test_performance():
    """اختبار الأداء"""
    print_header("Testing Performance")
    
    test_messages = [
        "مرحبا",
        "أريد المنيو",
        "كم سعر الكبسة؟",
        "أريد 2 هريس لحم",
        "وش طلبي؟"
    ]
    
    total_time = 0
    iterations = 100
    
    print(f"\n🏃 Running {iterations} iterations...")
    
    for _ in range(iterations):
        for message in test_messages:
            start = time.time()
            unified_classifier.classify(message, {})
            end = time.time()
            total_time += (end - start) * 1000  # Convert to ms
    
    avg_time = total_time / (iterations * len(test_messages))
    
    print(f"\n📊 Performance Results:")
    print(f"   Total classifications: {iterations * len(test_messages)}")
    print(f"   Total time: {total_time:.2f}ms")
    print(f"   Average time per classification: {avg_time:.2f}ms")
    
    if avg_time < 5:
        print("   ✅ Excellent performance!")
    elif avg_time < 10:
        print("   ✅ Good performance")
    elif avg_time < 20:
        print("   ⚠️ Acceptable performance")
    else:
        print("   ❌ Poor performance - optimization needed")


def test_dialects():
    """اختبار اللهجات المختلفة"""
    print_header("Testing Different Dialects")
    
    test_cases = [
        # Gulf dialect
        ("أبي كبسة", {}),
        ("وش عندكم؟", {}),
        
        # Egyptian dialect
        ("عايز شاورما", {}),
        ("ايه عندكو؟", {}),
        
        # Levantine dialect
        ("بدي حمص", {}),
        ("شو عندكن؟", {}),
        
        # Standard Arabic
        ("أريد طعام", {}),
        ("ماذا لديكم؟", {})
    ]
    
    for message, context in test_cases:
        result = unified_classifier.classify(message, context)
        print_result(message, result)


def run_all_tests():
    """تشغيل جميع الاختبارات"""
    print("\n" + "🚀" * 30)
    print("  Intent System Test Suite")
    print("🚀" * 30)
    
    # Initialize system
    print("\n📦 Initializing Intent System...")
    success = initialize_intent_system()
    
    if not success:
        print("❌ Failed to initialize intent system")
        return False
    
    print("✅ Intent System initialized successfully")
    
    # Get stats
    stats = unified_classifier.get_stats()
    print(f"\n📊 System Stats:")
    print(f"   - Rules loaded: {stats['total_rules']}")
    print(f"   - Intents covered: {stats['intents_covered']}")
    print(f"   - Keywords DB: {'✓' if stats['has_keywords_db'] else '✗'}")
    print(f"   - Examples DB: {'✓' if stats['has_examples_db'] else '✗'}")
    
    # Run tests
    try:
        test_greetings()
        test_confirmations()
        test_menu_requests()
        test_price_questions()
        test_order_food()
        test_cart_operations()
        test_special_patterns()
        test_context_awareness()
        test_dialects()
        test_performance()
        
        print_header("All Tests Completed! ✅")
        return True
        
    except Exception as e:
        print(f"\n❌ Error during tests: {str(e)}")
        import traceback
        traceback.print_exc()
        return False


if __name__ == "__main__":
    success = run_all_tests()
    sys.exit(0 if success else 1)

