#!/usr/bin/env python3
"""
Test Intent Guard - Food Name Detection
اختبار طبقة الحماية - اكتشاف أسماء الأصناف
"""

import os
import sys

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.intents.intent_guard import intent_guard
from app.intents.definitions import IntentType

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*70)
    print(f"  {text}")
    print("="*70)

def test_post_classify():
    """Test post_classify_guard with food names"""
    
    print_header("🧪 Testing Intent Guard - Food Name Detection")
    
    # Test cases: (message, classified_intent, expected_corrected_intent)
    test_cases = [
        # Should be corrected to ORDER_FOOD
        ("سمك", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("دجاج", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("هريس", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("كبسة", IntentType.ASK_PRICE, IntentType.ORDER_FOOD),
        ("برجر", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("عندكم سمك؟", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("عندكم دجاج؟", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("في هريس؟", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("موجود برجر؟", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("fish", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("chicken", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("do you have fish?", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("burger please", IntentType.ASK_MENU, IntentType.ORDER_FOOD),
        ("كم سعر السمك؟", IntentType.ASK_PRICE, IntentType.ORDER_FOOD),
        ("بكم الدجاج؟", IntentType.ASK_PRICE, IntentType.ORDER_FOOD),

        # Should NOT be corrected (no food names)
        ("المنيو", IntentType.ASK_MENU, IntentType.ASK_MENU),
        ("عندكم ايش؟", IntentType.ASK_MENU, IntentType.ASK_MENU),
        ("menu please", IntentType.ASK_MENU, IntentType.ASK_MENU),
        ("what do you have?", IntentType.ASK_MENU, IntentType.ASK_MENU),
        ("كم السعر؟", IntentType.ASK_PRICE, IntentType.ASK_PRICE),
    ]
    
    passed = 0
    failed = 0
    
    for message, classified_intent, expected_intent in test_cases:
        corrected_intent, was_corrected, reason = intent_guard.post_classify_guard(
            message, classified_intent
        )
        
        if corrected_intent == expected_intent:
            icon = "✅"
            passed += 1
        else:
            icon = "❌"
            failed += 1
        
        correction_text = f"(corrected)" if was_corrected else "(no change)"
        print(f"{icon} '{message}'")
        print(f"   Classified: {classified_intent.value}")
        print(f"   Expected: {expected_intent.value}")
        print(f"   Got: {corrected_intent.value} {correction_text}")
        if reason:
            print(f"   Reason: {reason}")
        print()
    
    # Summary
    print_header("📊 Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"❌ Failed: {failed}/{len(test_cases)}")
    print(f"Success Rate: {passed/len(test_cases)*100:.1f}%")
    
    if failed == 0:
        print("\n🎉 All tests passed!")
    else:
        print(f"\n⚠️  {failed} test(s) failed")

if __name__ == "__main__":
    test_post_classify()

