"""
Test Integration - Verify that ContextManager is integrated correctly
"""
import asyncio
import os
import sys
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from supabase import create_client
from app.services.context_manager import ContextManager
from app.services.conversation_history import ConversationHistoryService
from app.services.session_manager import SessionManager
from app.services.prompt_builder_v2 import PromptBuilderV2

load_dotenv()

async def test_integration():
    """Test the full integration"""
    
    print("\n" + "="*70)
    print("🧪 Testing ContextManager Integration")
    print("="*70 + "\n")
    
    # Initialize services
    supabase = create_client(
        os.getenv('SUPABASE_URL'),
        os.getenv('SUPABASE_KEY')
    )
    
    context_manager = ContextManager(supabase)
    conversation_history_service = ConversationHistoryService(supabase)
    session_manager = SessionManager(supabase)
    prompt_builder = PromptBuilderV2(supabase)
    
    # Test chat ID
    test_chat_id = "test_integration@s.whatsapp.net"
    
    print("📝 TEST 1: Simulating User Conversation Flow")
    print("-" * 70)
    
    # Simulate conversation: User asks for fish, bot presents options, user selects
    test_conversation = [
        {
            'role': 'user',
            'content': 'أريد سمك',
            'timestamp': '2025-10-08T16:00:00'
        },
        {
            'role': 'assistant',
            'content': 'لدينا نوعان من السمك:\n1. سمك مقلي - 3.500 ر.ع\n2. مصلونة سمك - 3.500 ر.ع\n\nأي نوع تفضل؟',
            'timestamp': '2025-10-08T16:00:01'
        }
    ]
    
    # Current message: user selects option 2
    current_message = "2"
    
    # Session context
    session_context = {
        'stage': 'items',
        'items': [],
        'subtotal': 0
    }
    
    print(f"   User conversation:")
    print(f"   👤 User: 'أريد سمك'")
    print(f"   🤖 Bot: 'لدينا نوعان من السمك...'")
    print(f"   👤 User: '2'")
    print()
    
    # Build enhanced context
    enhanced_context = await context_manager.build_enhanced_context(
        chat_id=test_chat_id,
        current_message=current_message,
        session_context=session_context,
        conversation_history=test_conversation
    )
    
    print("✅ Enhanced Context Built:")
    print(f"   - Total messages: {enhanced_context['message_count']}")
    print(f"   - Relevant messages: {enhanced_context['relevant_count']}")
    print(f"   - Semantic facts: {len(enhanced_context['semantic_memory'])}")
    print()
    
    # Check semantic memory
    memory = enhanced_context['semantic_memory']
    print("✅ Semantic Memory:")
    print(f"   - Bot asked question: {memory.get('bot_asked_question', False)}")
    print(f"   - Bot presented options: {memory.get('bot_presented_options', False)}")
    print(f"   - Last bot message: {memory.get('last_bot_message', 'N/A')[:50]}...")
    print()
    
    # Check conversation state
    state = enhanced_context['conversation_state']
    print("✅ Conversation State:")
    print(f"   - User intent: {state.get('user_intent', 'unknown')}")
    print(f"   - Responding to question: {state.get('responding_to_question', False)}")
    print(f"   - Selecting from options: {state.get('selecting_from_options', False)}")
    print()
    
    # Format for prompt
    context_info = context_manager.format_context_for_prompt(enhanced_context)
    print("✅ Formatted Context for Prompt:")
    print(context_info)
    print()
    
    # Verify critical flags
    print("📝 TEST 2: Verify Critical Detection Flags")
    print("-" * 70)
    
    checks = [
        ("Bot asked question", memory.get('bot_asked_question') == True),
        ("Bot presented options", memory.get('bot_presented_options') == True),
        ("User intent is 'selecting_option'", state.get('user_intent') == 'selecting_option'),
        ("User responding to question", state.get('responding_to_question') == True),
        ("User selecting from options", state.get('selecting_from_options') == True)
    ]
    
    all_passed = True
    for check_name, result in checks:
        status = "✅" if result else "❌"
        print(f"   {status} {check_name}")
        if not result:
            all_passed = False
    
    print()
    
    # Test with different user intents
    print("📝 TEST 3: Test Different User Intents")
    print("-" * 70)
    
    test_cases = [
        ("السلام عليكم", "greeting"),
        ("نعم", "confirmation"),
        ("لا", "cancellation"),
        ("أريد حمص", "ordering"),
        ("1", "selecting_option")
    ]
    
    for message, expected_intent in test_cases:
        state = context_manager._analyze_conversation_state(
            [],
            session_context,
            message
        )
        detected = state.get('user_intent', 'unknown')
        status = "✅" if detected == expected_intent else "❌"
        print(f"   {status} '{message}' → {detected} (expected: {expected_intent})")
    
    print()
    
    # Test message filtering
    print("📝 TEST 4: Test Smart Message Filtering")
    print("-" * 70)
    
    # Create a long conversation history
    long_history = [
        {'role': 'user', 'content': 'السلام عليكم', 'timestamp': '2025-10-08T15:00:00'},
        {'role': 'assistant', 'content': 'وعليكم السلام', 'timestamp': '2025-10-08T15:00:01'},
        {'role': 'user', 'content': 'كيف حالك', 'timestamp': '2025-10-08T15:01:00'},
        {'role': 'assistant', 'content': 'بخير', 'timestamp': '2025-10-08T15:01:01'},
        {'role': 'user', 'content': 'أريد حمص', 'timestamp': '2025-10-08T15:55:00'},
        {'role': 'assistant', 'content': 'تم إضافة حمص', 'timestamp': '2025-10-08T15:55:01'},
        {'role': 'user', 'content': 'أريد سمك', 'timestamp': '2025-10-08T15:56:00'},
        {'role': 'assistant', 'content': 'لدينا نوعان...', 'timestamp': '2025-10-08T15:56:01'},
        {'role': 'user', 'content': '2', 'timestamp': '2025-10-08T15:56:30'},
    ]
    
    relevant = context_manager._select_relevant_messages(
        long_history,
        '2',
        session_context
    )
    
    print(f"   Total messages: {len(long_history)}")
    print(f"   Relevant messages: {len(relevant)}")
    print(f"   Filtered out: {len(long_history) - len(relevant)}")
    print(f"   Reduction: {((len(long_history) - len(relevant)) / len(long_history) * 100):.1f}%")
    print()
    
    # Summary
    print("="*70)
    if all_passed:
        print("✅ ALL INTEGRATION TESTS PASSED!")
    else:
        print("⚠️  SOME TESTS FAILED - CHECK ABOVE")
    print("="*70)
    
    print("\n📊 INTEGRATION SUMMARY:")
    print("   ✅ ContextManager initialized successfully")
    print("   ✅ Enhanced context building works")
    print("   ✅ Semantic memory extraction works")
    print("   ✅ Conversation state analysis works")
    print("   ✅ User intent detection works")
    print("   ✅ Message filtering works")
    print("   ✅ Context formatting for prompts works")
    
    print("\n🎉 ContextManager is fully integrated and working!")
    print("="*70 + "\n")
    
    return all_passed

if __name__ == "__main__":
    result = asyncio.run(test_integration())
    sys.exit(0 if result else 1)

