#!/usr/bin/env python3
"""
Test Full Search Flow
اختبار التدفق الكامل للبحث
"""

import os
import sys
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.services.menu_search_service import menu_search_service

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

def print_results(search_result: dict):
    """Print search results"""
    results = search_result.get('results', [])
    corrected_query = search_result.get('corrected_query', '')
    search_methods = search_result.get('search_methods', [])
    
    print(f"\n📊 Search Summary:")
    print(f"   Final Query: '{corrected_query}'")
    print(f"   Methods: {', '.join(search_methods)}")
    print(f"   Total: {len(results)} items")
    
    if results:
        print(f"\n📋 Results:")
        for i, item in enumerate(results, 1):
            name_ar = item.get('name_ar', 'N/A')
            name_en = item.get('name_en', 'N/A')
            match_type = item.get('match_type', 'unknown')
            match_score = item.get('match_score', 0)
            
            if match_type == 'exact':
                icon = '🎯'
            elif match_type == 'partial':
                icon = '🔍'
            else:
                icon = '🧠'
            
            print(f"{icon} {i}. {name_ar} / {name_en}")
            print(f"   {match_type} ({match_score:.2f})")

def test_with_question_words():
    """Test queries with question words"""
    print_header("🧪 Test 1: Queries with Question Words")
    
    test_cases = [
        "عندكم سمك؟",
        "هل فيه هريس دجاج؟",
        "ابي كبسة عوان لو سمحت",
        "do you have fish?",
        "can i get chicken harees?",
    ]
    
    for query in test_cases:
        print(f"\n🔍 Original Query: '{query}'")
        result = menu_search_service.search_menu(query, limit=3)
        print_results(result)

def test_with_spelling_errors():
    """Test queries with spelling errors"""
    print_header("🧪 Test 2: Queries with Spelling Errors")
    
    test_cases = [
        "عندكم سمج؟",  # سمج → سمك
        "ابي هريص دجاچ",  # هريص → هريس, دجاچ → دجاج
        "فيه كبصة؟",  # كبصة → كبسة
    ]
    
    for query in test_cases:
        print(f"\n🔍 Original Query: '{query}'")
        result = menu_search_service.search_menu(query, limit=3)
        print_results(result)

def test_complex_queries():
    """Test complex queries"""
    print_header("🧪 Test 3: Complex Queries")
    
    test_cases = [
        "هل عندكم سمج مقلى مع رز قبولي؟",  # cleaning + correction
        "ابي هريص دجاچ لو سمحت",  # cleaning + correction
        "do you have fried fich?",  # cleaning + correction
    ]
    
    for query in test_cases:
        print(f"\n🔍 Original Query: '{query}'")
        result = menu_search_service.search_menu(query, limit=5)
        print_results(result)

def test_clean_queries():
    """Test already clean queries"""
    print_header("🧪 Test 4: Clean Queries (No Processing Needed)")
    
    test_cases = [
        "سمك",
        "هريس دجاج",
        "كبسة عوان",
        "fish",
        "chicken harees",
    ]
    
    for query in test_cases:
        print(f"\n🔍 Query: '{query}'")
        result = menu_search_service.search_menu(query, limit=3)
        print_results(result)

def main():
    """Run all tests"""
    print_header("🚀 Full Search Flow Tests")
    print("Testing: Text Cleaning → Word Correction → Search (Exact/Partial/Semantic)")
    
    try:
        test_with_question_words()
        test_with_spelling_errors()
        test_complex_queries()
        test_clean_queries()
        
        print_header("🎉 All Tests Completed!")
        
    except Exception as e:
        print(f"\n❌ Test failed with error: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    main()

