#!/usr/bin/env python3
"""
Test FAISS Menu Search Improvements
اختبار تحسينات FAISS
"""

import os
import sys
import asyncio
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.services.faiss_menu_search import faiss_menu_search
from app.database.supabase_client import get_supabase_client

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

async def load_menu():
    """Load menu items from Supabase"""
    print_header("📥 Loading Menu from Supabase")
    
    supabase = get_supabase_client()
    menu_items = supabase.get_all_menu_items(active_only=True)
    
    print(f"✅ Loaded {len(menu_items)} menu items")
    
    # Load into FAISS
    await faiss_menu_search.load_menu(menu_items)
    print(f"✅ FAISS index built successfully")

async def test_exact_match_boost():
    """Test exact match boost feature"""
    print_header("🧪 Test 1: Exact Match Boost")
    
    test_cases = [
        "سمك",
        "هريس",
        "كبسة",
        "fish",
        "harees"
    ]
    
    for query in test_cases:
        print(f"\n🔍 Searching for: '{query}'")
        
        # Search with boost
        results_with_boost = await faiss_menu_search.search(
            query, 
            top_k=5, 
            min_score=0.3,
            exact_match_boost=True
        )
        
        if results_with_boost:
            print(f"✅ Found {len(results_with_boost)} results (with boost):")
            for i, (item, score) in enumerate(results_with_boost, 1):
                name_ar = item.get('name_ar', 'N/A')
                name_en = item.get('name_en', 'N/A')
                print(f"   {i}. {name_ar} / {name_en} ({score:.1f}%)")
        else:
            print("❌ No results")

async def test_top_k_adjustment():
    """Test top_k parameter adjustment"""
    print_header("🧪 Test 2: Top-K Adjustment")
    
    query = "شوربة"
    
    for top_k in [3, 5, 10]:
        print(f"\n🔍 Searching for: '{query}' (top_k={top_k})")
        
        results = await faiss_menu_search.search(
            query, 
            top_k=top_k, 
            min_score=0.3
        )
        
        print(f"✅ Found {len(results)} results:")
        for i, (item, score) in enumerate(results, 1):
            name_ar = item.get('name_ar', 'N/A')
            print(f"   {i}. {name_ar} ({score:.1f}%)")

async def test_keyword_filter():
    """Test keyword filter with exact match"""
    print_header("🧪 Test 3: Keyword Filter + Exact Match")
    
    test_cases = [
        ("سمك", "Should find fish items"),
        ("دجاج", "Should find chicken items"),
        ("لحم", "Should find meat items"),
    ]
    
    for query, description in test_cases:
        print(f"\n🔍 Testing: {description}")
        print(f"   Query: '{query}'")
        
        # With keyword filter
        results_filtered = await faiss_menu_search.search(
            query, 
            top_k=5, 
            min_score=0.3,
            keyword_filter=True,
            exact_match_boost=True
        )
        
        if results_filtered:
            print(f"   ✅ Found {len(results_filtered)} filtered results:")
            for item, score in results_filtered:
                name_ar = item.get('name_ar', 'N/A')
                print(f"      - {name_ar} ({score:.1f}%)")
        else:
            print(f"   ❌ No results")

async def test_comparison():
    """Compare results with and without improvements"""
    print_header("🧪 Test 4: Before/After Comparison")
    
    query = "سمك"
    
    print(f"\n🔍 Query: '{query}'")
    
    # Without boost
    print("\n📊 Without exact match boost:")
    results_no_boost = await faiss_menu_search.search(
        query, 
        top_k=5, 
        min_score=0.3,
        exact_match_boost=False
    )
    
    for i, (item, score) in enumerate(results_no_boost, 1):
        name_ar = item.get('name_ar', 'N/A')
        print(f"   {i}. {name_ar} ({score:.1f}%)")
    
    # With boost
    print("\n📊 With exact match boost:")
    results_with_boost = await faiss_menu_search.search(
        query, 
        top_k=5, 
        min_score=0.3,
        exact_match_boost=True
    )
    
    for i, (item, score) in enumerate(results_with_boost, 1):
        name_ar = item.get('name_ar', 'N/A')
        print(f"   {i}. {name_ar} ({score:.1f}%)")

async def main():
    """Run all tests"""
    print_header("🚀 FAISS Menu Search Improvements Tests")
    
    try:
        # Load menu
        await load_menu()
        
        # Run tests
        await test_exact_match_boost()
        await test_top_k_adjustment()
        await test_keyword_filter()
        await test_comparison()
        
        print_header("🎉 All Tests Completed!")
        
    except Exception as e:
        print(f"\n❌ Test failed with error: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    asyncio.run(main())

