#!/usr/bin/env python3
"""
Test Exact Gate
اختبار بوابة التطابق التام
"""

import os
import sys
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.services.menu_search_service import menu_search_service

def print_header(text: str):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)

def print_results(query: str, result: dict):
    """Print search results"""
    print(f"\n🔍 Query: '{query}'")
    print(f"📊 Found {result['total_found']} results")
    print(f"Methods: {', '.join(result['search_methods'])}")
    print(f"Exact: {result['exact_count']}, Partial: {result['partial_count']}, Semantic: {result['semantic_count']}")
    
    if result['results']:
        print("\nResults:")
        for i, item in enumerate(result['results'], 1):
            name_ar = item['name_ar']
            match_type = item.get('match_type', 'unknown')
            match_score = item.get('match_score', 0)
            print(f"  {i}. {name_ar} ({match_type}, score={match_score:.2f})")
    else:
        print("  No results found")

def test_exact_gate():
    """Test Exact Gate functionality"""
    print_header("🧪 Test Exact Gate")
    
    test_cases = [
        ("هريس", ["هريس دجاج", "هريس لحم"]),
        ("عندكم هريس؟", ["هريس دجاج", "هريس لحم"]),
        ("شوربة هريس", ["شوربة هريس باللحم"]),
        ("سمك", ["سمك مقلي"]),
        ("كبسة", ["كبسة عوان"]),
    ]
    
    passed = 0
    failed = 0
    
    for query, expected_items in test_cases:
        result = menu_search_service.search_menu(query, limit=5)
        print_results(query, result)
        
        # Check if results match expected
        actual_items = [item['name_ar'] for item in result['results']]
        
        # Check if all expected items are in results
        all_found = all(any(exp in actual for actual in actual_items) for exp in expected_items)
        
        if all_found:
            print(f"✅ All expected items found")
            passed += 1
        else:
            print(f"❌ Some expected items missing")
            print(f"   Expected: {expected_items}")
            print(f"   Got: {actual_items}")
            failed += 1
    
    print_header("📊 Summary")
    print(f"✅ Passed: {passed}/{len(test_cases)}")
    print(f"❌ Failed: {failed}/{len(test_cases)}")
    
    return passed, failed

def test_category_guard():
    """Test Category Guard functionality"""
    print_header("🧪 Test Category Guard")
    
    test_cases = [
        ("هريس", "Should NOT include soups/salads/offers"),
        ("شوربة", "Should include soups"),
        ("سلطة", "Should include salads"),
    ]
    
    for query, description in test_cases:
        result = menu_search_service.search_menu(query, limit=10)
        print_results(query, result)
        print(f"Description: {description}")
        
        # Check categories
        categories = [item.get('category', 'N/A') for item in result['results']]
        print(f"Categories: {set(categories)}")

def main():
    """Run all tests"""
    print_header("🚀 Exact Gate Tests")
    
    passed, failed = test_exact_gate()
    test_category_guard()
    
    print_header("🎯 Final Summary")
    if failed == 0:
        print("🎉 All tests passed!")
    else:
        print(f"⚠️  {failed} test(s) failed")

if __name__ == "__main__":
    main()

