"""
Test Evolution API connection and webhook
"""
import os
import sys
from pathlib import Path
from dotenv import load_dotenv
import requests

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

EVOLUTION_API_URL = os.getenv('EVOLUTION_API_URL')
EVOLUTION_API_KEY = os.getenv('EVOLUTION_API_KEY')
EVOLUTION_INSTANCE_NAME = os.getenv('EVOLUTION_INSTANCE_NAME')

print("="*70)
print("🔍 Testing Evolution API Connection")
print("="*70)
print(f"\n📡 Evolution API URL: {EVOLUTION_API_URL}")
print(f"🔑 API Key: {EVOLUTION_API_KEY[:20]}...")
print(f"📱 Instance Name: {EVOLUTION_INSTANCE_NAME}")
print()

headers = {
    "apikey": EVOLUTION_API_KEY,
    "Content-Type": "application/json"
}

# Test 1: Check instance status
print("="*70)
print("Test 1: Check Instance Status")
print("="*70)

try:
    url = f"{EVOLUTION_API_URL}/instance/connectionState/{EVOLUTION_INSTANCE_NAME}"
    response = requests.get(url, headers=headers, timeout=10)
    
    print(f"Status Code: {response.status_code}")
    
    if response.status_code == 200:
        data = response.json()
        print(f"✅ Instance Status: {data}")
        
        state = data.get('instance', {}).get('state')
        if state == 'open':
            print("✅ Instance is CONNECTED!")
        else:
            print(f"⚠️ Instance state: {state}")
    else:
        print(f"❌ Error: {response.text}")
except Exception as e:
    print(f"❌ Error: {e}")

print()

# Test 2: Check webhook settings
print("="*70)
print("Test 2: Check Webhook Settings")
print("="*70)

try:
    url = f"{EVOLUTION_API_URL}/webhook/find/{EVOLUTION_INSTANCE_NAME}"
    response = requests.get(url, headers=headers, timeout=10)
    
    print(f"Status Code: {response.status_code}")
    
    if response.status_code == 200:
        data = response.json()
        print(f"✅ Webhook Settings:")
        print(f"   URL: {data.get('webhook', {}).get('url', 'Not set')}")
        print(f"   Enabled: {data.get('webhook', {}).get('enabled', False)}")
        print(f"   Events: {data.get('webhook', {}).get('events', [])}")
        
        webhook_url = data.get('webhook', {}).get('url', '')
        if 'ngrok' in webhook_url or 'webhook' in webhook_url:
            print("✅ Webhook URL looks correct!")
        else:
            print("⚠️ Webhook URL might be incorrect")
    else:
        print(f"❌ Error: {response.text}")
except Exception as e:
    print(f"❌ Error: {e}")

print()

# Test 3: Get ngrok URL
print("="*70)
print("Test 3: Check ngrok URL")
print("="*70)

try:
    response = requests.get("http://localhost:4040/api/tunnels", timeout=5)
    if response.status_code == 200:
        data = response.json()
        if data.get('tunnels'):
            ngrok_url = data['tunnels'][0]['public_url']
            print(f"✅ ngrok URL: {ngrok_url}")
            print(f"✅ Expected Webhook: {ngrok_url}/webhook/whatsapp")
        else:
            print("❌ No ngrok tunnels found")
    else:
        print("❌ ngrok not running")
except Exception as e:
    print(f"❌ ngrok not accessible: {e}")

print()
print("="*70)
print("📋 Summary")
print("="*70)
print("""
✅ Next Steps:
1. Make sure Instance is CONNECTED in Evolution API Dashboard
2. Make sure Webhook URL is set to: https://your-ngrok-url.ngrok-free.dev/webhook/whatsapp
3. Make sure Webhook Events include: messages.upsert
4. Send a test message from another WhatsApp number
5. Check logs: tail -f /tmp/server.log
""")

