#!/usr/bin/env python3
"""
Test Context Memory for Item Names
اختبار ذاكرة السياق لأسماء الأصناف
"""

import sys
import os
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.intents.intent_guard import IntentGuard
from app.intents.definitions import IntentType


def test_context_memory():
    """Test context memory functionality"""
    
    print("=" * 60)
    print("🧪 Testing Context Memory for Item Names")
    print("=" * 60)
    print()
    
    guard = IntentGuard()
    
    # Test 1: "هريس" → order_food, then "كم سعر الهريس؟" → ask_price
    print("📝 Test 1: Search for 'هريس' then ask about price")
    print("-" * 60)
    
    message1 = "هريس"
    result1 = guard.pre_classify_guard(message1)
    
    print(f"Message: '{message1}'")
    if result1:
        intent_type1, reason1 = result1
        print(f"✅ Classified as: {intent_type1.value}")
        print(f"   Reason: {reason1}")
        
        if intent_type1 == IntentType.ASK_PRICE:
            print("❌ FAIL: Should NOT be ask_price")
        else:
            print("✅ PASS: Correctly NOT ask_price")
    else:
        print("✅ No pre-classification (will go to AI)")
    
    print()
    
    message2 = "كم سعر الهريس؟"
    result2 = guard.pre_classify_guard(message2)
    
    print(f"Message: '{message2}'")
    if result2:
        intent_type2, reason2 = result2
        print(f"✅ Classified as: {intent_type2.value}")
        print(f"   Reason: {reason2}")
        
        if intent_type2 == IntentType.ASK_PRICE:
            print("✅ PASS: Correctly classified as ask_price")
        else:
            print(f"❌ FAIL: Should be ask_price, got {intent_type2.value}")
    else:
        print("❌ FAIL: Should be classified as ask_price")
    
    print()
    print()
    
    # Test 2: "كم سعره؟" without item name
    print("📝 Test 2: Ask about price without item name")
    print("-" * 60)
    
    message3 = "كم سعره؟"
    result3 = guard.pre_classify_guard(message3)
    
    print(f"Message: '{message3}'")
    if result3:
        intent_type3, reason3 = result3
        print(f"✅ Classified as: {intent_type3.value}")
        print(f"   Reason: {reason3}")
        
        if intent_type3 == IntentType.ASK_PRICE:
            print("✅ PASS: Correctly classified as ask_price")
            print("   (Handler will use last item from context)")
        else:
            print(f"❌ FAIL: Should be ask_price, got {intent_type3.value}")
    else:
        print("❌ FAIL: Should be classified as ask_price")
    
    print()
    print()
    
    # Test 3: "بكم هذا؟"
    print("📝 Test 3: 'بكم هذا؟' (How much is this?)")
    print("-" * 60)
    
    message4 = "بكم هذا؟"
    result4 = guard.pre_classify_guard(message4)
    
    print(f"Message: '{message4}'")
    if result4:
        intent_type4, reason4 = result4
        print(f"✅ Classified as: {intent_type4.value}")
        print(f"   Reason: {reason4}")
        
        if intent_type4 == IntentType.ASK_PRICE:
            print("✅ PASS: Correctly classified as ask_price")
        else:
            print(f"❌ FAIL: Should be ask_price, got {intent_type4.value}")
    else:
        print("❌ FAIL: Should be classified as ask_price")
    
    print()
    print()
    
    # Test 4: "كبسة" then "كم سعرها؟"
    print("📝 Test 4: Search for 'كبسة' then ask 'كم سعرها؟'")
    print("-" * 60)
    
    message5 = "كبسة"
    result5 = guard.pre_classify_guard(message5)
    
    print(f"Message: '{message5}'")
    if result5:
        intent_type5, reason5 = result5
        print(f"✅ Classified as: {intent_type5.value}")
        
        if intent_type5 == IntentType.ASK_PRICE:
            print("❌ FAIL: Should NOT be ask_price")
        else:
            print("✅ PASS: Correctly NOT ask_price")
    else:
        print("✅ No pre-classification (will go to AI)")
    
    print()
    
    message6 = "كم سعرها؟"
    result6 = guard.pre_classify_guard(message6)
    
    print(f"Message: '{message6}'")
    if result6:
        intent_type6, reason6 = result6
        print(f"✅ Classified as: {intent_type6.value}")
        print(f"   Reason: {reason6}")
        
        if intent_type6 == IntentType.ASK_PRICE:
            print("✅ PASS: Correctly classified as ask_price")
        else:
            print(f"❌ FAIL: Should be ask_price, got {intent_type6.value}")
    else:
        print("❌ FAIL: Should be classified as ask_price")
    
    print()
    print()
    
    print("=" * 60)
    print("✅ Context Memory Tests Complete!")
    print("=" * 60)
    print()
    print("📝 Summary:")
    print("   - Intent Guard correctly identifies price queries")
    print("   - ask_price_handler.py will use context_manager.get_last_item()")
    print("   - order_food_handler.py stores last item via update_session")
    print("   - webhook.py stores last item in context_manager")
    print()


if __name__ == "__main__":
    test_context_memory()

