#!/usr/bin/env python3
"""
Test Cache-Based Search
========================
Tests the new cache-based menu search system.

Usage:
    python scripts/test_cache_search.py
"""

import sys
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.services.menu_search_service import MenuSearchService
from app.utils.text_helpers import clean_user_query


def print_header(text):
    """Print formatted header"""
    print("\n" + "="*80)
    print(f"  {text}")
    print("="*80)


def test_search(query: str):
    """Test search with a query"""
    print(f"\n🔍 Query: '{query}'")
    
    # Clean query
    core_term = clean_user_query(query)
    print(f"   Core term: '{core_term}'")
    
    # Search
    search_service = MenuSearchService()
    result = search_service.search_menu(core_term, limit=5)
    
    # Display results
    print(f"\n📊 Results: {result['total_found']} items found")
    print(f"   - Exact: {result['exact_count']}")
    print(f"   - Partial: {result['partial_count']}")
    print(f"   - Semantic: {result['semantic_count']}")
    
    if result['results']:
        print(f"\n📋 Top {len(result['results'])} results:")
        for i, item in enumerate(result['results'], 1):
            match_type = item.get('match_type', 'unknown')
            match_score = item.get('match_score', 0)
            print(f"   {i}. {item['name_ar']} ({match_type}, score: {match_score:.2f})")
    else:
        print("\n❌ No results found")


def main():
    """Run all tests"""
    print_header("🧪 Testing Cache-Based Menu Search")
    
    test_cases = [
        "هريس",
        "عندكم هريس",
        "شوربة هريس",
        "سمك",
        "كبسة",
        "مشوي",
    ]
    
    for query in test_cases:
        test_search(query)
    
    print_header("✅ All Tests Complete")


if __name__ == "__main__":
    main()

