#!/usr/bin/env python3
"""
Full Menu Synchronization Script
Syncs menu data: Supabase → Cache → FAISS

Usage:
    python3 scripts/sync_menu_full.py [--force] [--validate] [--status]

Options:
    --force      Force sync even if cache is fresh
    --validate   Validate sync after completion
    --status     Show sync status only (no sync)
"""
import sys
import os
import argparse
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

# Load environment variables
from dotenv import load_dotenv
load_dotenv()

from app.services.menu_sync_service import get_menu_sync_service
from app.utils.logger import logger
from datetime import datetime
import json


def print_banner():
    """Print script banner"""
    print("="*80)
    print("🔄 MENU SYNCHRONIZATION SCRIPT")
    print("="*80)
    print(f"📅 Started at: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print("="*80)
    print()


def print_status(status: dict):
    """Print sync status"""
    print("\n📊 SYNC STATUS:")
    print("="*80)
    
    # Supabase
    print("\n🗄️  Supabase:")
    supabase = status.get('supabase', {})
    print(f"  Status: {'✅ Healthy' if supabase.get('healthy') else '❌ Unhealthy'}")
    print(f"  Items: {supabase.get('item_count', 0)}")
    
    # Cache
    print("\n💾 Cache:")
    cache = status.get('cache', {})
    print(f"  Loaded: {'✅ Yes' if cache.get('loaded') else '❌ No'}")
    print(f"  Items: {cache.get('item_count', 0)}")
    print(f"  Categories: {cache.get('category_count', 0)}")
    print(f"  Last Update: {cache.get('last_update', 'Never')}")
    print(f"  Stale: {'⚠️  Yes' if cache.get('is_stale') else '✅ No'}")
    
    # FAISS
    print("\n🔍 FAISS Index:")
    faiss = status.get('faiss', {})
    print(f"  Exists: {'✅ Yes' if faiss.get('exists') else '❌ No'}")
    if faiss.get('exists'):
        size_mb = faiss.get('size_bytes', 0) / 1024 / 1024
        print(f"  Size: {size_mb:.2f} MB")
    print(f"  Path: {faiss.get('path', 'N/A')}")
    
    # Sync needed
    print(f"\n🔄 Sync Needed: {'⚠️  Yes' if status.get('sync_needed') else '✅ No'}")
    print("="*80)


def print_result(result: dict):
    """Print sync result"""
    print("\n📋 SYNC RESULT:")
    print("="*80)
    
    success = result.get('success', False)
    print(f"\nStatus: {'✅ SUCCESS' if success else '❌ FAILED'}")
    
    # Steps
    steps = result.get('steps', {})
    if steps:
        print("\n📝 Steps Completed:")
        for step, status in steps.items():
            print(f"  ✅ {step}: {status}")
    
    # Errors
    errors = result.get('errors', [])
    if errors:
        print("\n❌ Errors:")
        for error in errors:
            print(f"  - {error}")
    
    # Item count
    if 'item_count' in result:
        print(f"\n📊 Total Items Synced: {result['item_count']}")
    
    print(f"\n⏱️  Timestamp: {result.get('timestamp', 'N/A')}")
    print("="*80)


def print_validation(validation: dict):
    """Print validation result"""
    print("\n🔍 VALIDATION RESULT:")
    print("="*80)
    
    valid = validation.get('valid', False)
    print(f"\nStatus: {'✅ VALID' if valid else '❌ INVALID'}")
    
    # Issues
    issues = validation.get('issues', [])
    if issues:
        print("\n❌ Issues Found:")
        for issue in issues:
            print(f"  - {issue}")
    
    # Warnings
    warnings = validation.get('warnings', [])
    if warnings:
        print("\n⚠️  Warnings:")
        for warning in warnings:
            print(f"  - {warning}")
    
    if valid and not warnings:
        print("\n✅ All components are in sync!")
    
    print("="*80)


def main():
    """Main function"""
    parser = argparse.ArgumentParser(
        description='Full menu synchronization: Supabase → Cache → FAISS'
    )
    parser.add_argument(
        '--force',
        action='store_true',
        help='Force sync even if cache is fresh'
    )
    parser.add_argument(
        '--validate',
        action='store_true',
        help='Validate sync after completion'
    )
    parser.add_argument(
        '--status',
        action='store_true',
        help='Show sync status only (no sync)'
    )
    parser.add_argument(
        '--json',
        action='store_true',
        help='Output in JSON format'
    )
    
    args = parser.parse_args()
    
    try:
        # Initialize service
        sync_service = get_menu_sync_service()
        
        if not args.json:
            print_banner()
        
        # Status only mode
        if args.status:
            status = sync_service.get_sync_status()
            
            if args.json:
                print(json.dumps(status, indent=2))
            else:
                print_status(status)
            
            return 0
        
        # Perform sync
        if not args.json:
            print("🚀 Starting synchronization...")
            if args.force:
                print("⚡ Force mode enabled")
            print()
        
        result = sync_service.sync_full(force=args.force)
        
        if args.json:
            print(json.dumps(result, indent=2))
        else:
            print_result(result)
        
        # Validate if requested
        if args.validate and result.get('success'):
            if not args.json:
                print("\n🔍 Validating sync...")
            
            validation = sync_service.validate_sync()
            
            if args.json:
                print(json.dumps(validation, indent=2))
            else:
                print_validation(validation)
            
            if not validation.get('valid'):
                return 1
        
        # Return exit code
        return 0 if result.get('success') else 1
        
    except KeyboardInterrupt:
        print("\n\n⚠️  Sync interrupted by user")
        return 130
    except Exception as e:
        if args.json:
            print(json.dumps({'error': str(e)}, indent=2))
        else:
            print(f"\n❌ Error: {str(e)}")
            logger.exception("Sync failed")
        return 1


if __name__ == '__main__':
    exit_code = main()
    sys.exit(exit_code)

