"""
Setup Webhook Script
Registers webhook with Evolution API
"""
import sys
import os
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

import httpx
from app.config import settings

def setup_webhook(webhook_url: str):
    """Setup webhook in Evolution API"""
    print("\n" + "="*80)
    print("🔧 SETTING UP WEBHOOK")
    print("="*80)
    
    print(f"\n📡 Evolution API: {settings.EVOLUTION_API_URL}")
    print(f"🔑 Instance: {settings.EVOLUTION_INSTANCE_NAME}")
    print(f"🔗 Webhook URL: {webhook_url}")
    
    try:
        # Set webhook
        url = f"{settings.EVOLUTION_API_URL}/webhook/set/{settings.EVOLUTION_INSTANCE_NAME}"
        headers = {
            "apikey": settings.EVOLUTION_API_KEY,
            "Content-Type": "application/json"
        }
        
        payload = {
            "url": webhook_url,
            "webhook_by_events": False,
            "webhook_base64": False,
            "events": [
                "MESSAGES_UPSERT",
                "MESSAGES_UPDATE",
                "MESSAGES_DELETE",
                "SEND_MESSAGE",
                "CONNECTION_UPDATE"
            ]
        }
        
        print(f"\n📤 Sending webhook configuration...")
        print(f"   Events: {', '.join(payload['events'])}")
        
        response = httpx.post(url, headers=headers, json=payload, timeout=10)
        
        if response.status_code == 200 or response.status_code == 201:
            print(f"\n✅ Webhook configured successfully!")
            print(f"   Response: {response.json()}")
            return True
        else:
            print(f"\n❌ Failed to configure webhook (status {response.status_code})")
            print(f"   Response: {response.text}")
            return False
            
    except Exception as e:
        print(f"\n❌ Error setting up webhook: {e}")
        return False


def verify_webhook():
    """Verify webhook configuration"""
    print("\n" + "="*80)
    print("🔍 VERIFYING WEBHOOK")
    print("="*80)
    
    try:
        url = f"{settings.EVOLUTION_API_URL}/webhook/find/{settings.EVOLUTION_INSTANCE_NAME}"
        headers = {"apikey": settings.EVOLUTION_API_KEY}
        
        print(f"\n🔗 Fetching webhook configuration...")
        response = httpx.get(url, headers=headers, timeout=10)
        
        if response.status_code == 200:
            data = response.json()
            webhook_url = data.get('webhook', {}).get('url', '')
            events = data.get('webhook', {}).get('events', [])
            
            print(f"\n✅ Webhook configuration:")
            print(f"   URL: {webhook_url}")
            print(f"   Events: {events}")
            
            if webhook_url and 'webhook/whatsapp' in webhook_url:
                print(f"\n✅ Webhook is configured correctly!")
                return True
            else:
                print(f"\n⚠️  Webhook URL might be incorrect")
                return False
        else:
            print(f"\n⚠️  Could not verify webhook (status {response.status_code})")
            return False
            
    except Exception as e:
        print(f"\n❌ Error verifying webhook: {e}")
        return False


def main():
    """Main function"""
    print("\n" + "="*80)
    print("🚀 WEBHOOK SETUP TOOL")
    print("="*80)
    
    # Ask for webhook URL
    print("\n📝 Please provide your webhook URL")
    print("   Format: http://your-server-ip:8000/webhook/whatsapp")
    print("   Or: https://your-domain.com/webhook/whatsapp")
    print("\n   Examples:")
    print("   - http://192.168.1.100:8000/webhook/whatsapp (local network)")
    print("   - https://myserver.com/webhook/whatsapp (public domain)")
    print("   - http://localhost:8000/webhook/whatsapp (testing only)")
    
    webhook_url = input("\n🔗 Enter webhook URL: ").strip()
    
    if not webhook_url:
        print("\n❌ No URL provided. Exiting.")
        return
    
    if not webhook_url.startswith(('http://', 'https://')):
        print("\n❌ Invalid URL. Must start with http:// or https://")
        return
    
    if not webhook_url.endswith('/webhook/whatsapp'):
        print("\n⚠️  Warning: URL should end with /webhook/whatsapp")
        confirm = input("   Continue anyway? (y/n): ").strip().lower()
        if confirm != 'y':
            print("\n❌ Cancelled.")
            return
    
    # Setup webhook
    success = setup_webhook(webhook_url)
    
    if success:
        # Verify
        verify_webhook()
        
        print("\n" + "="*80)
        print("🎉 WEBHOOK SETUP COMPLETE!")
        print("="*80)
        print("\n✅ Your chatbot is now ready to receive messages!")
        print("\n📱 Test it by sending a message to your WhatsApp number")
        print("   Example: 'السلام عليكم' or 'Hello'")
        print("\n📊 Monitor logs:")
        print("   tail -f logs/app.log")
    else:
        print("\n" + "="*80)
        print("❌ WEBHOOK SETUP FAILED")
        print("="*80)
        print("\n📋 Troubleshooting:")
        print("   1. Check Evolution API URL and API key in .env")
        print("   2. Verify instance name is correct")
        print("   3. Make sure Evolution API can reach your webhook URL")
        print("   4. Check firewall settings")


if __name__ == "__main__":
    main()

