"""
Remove prompt_sections table from database
Since all prompts are now in app/prompts/*.py files, this table is no longer needed
"""
import os
import sys
import json
from pathlib import Path
from datetime import datetime

sys.path.insert(0, str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from supabase import create_client

load_dotenv()

def main():
    print("\n" + "="*70)
    print("🧹 Remove prompt_sections Table")
    print("="*70 + "\n")
    
    supabase = create_client(
        os.getenv('SUPABASE_URL'),
        os.getenv('SUPABASE_KEY')
    )
    
    # Step 1: Create backup
    print("📦 Step 1: Creating backup...")
    try:
        result = supabase.table('prompt_sections').select('*').execute()
        sections = result.data if result.data else []
        
        if sections:
            print(f"   ✅ Found {len(sections)} sections")
            
            # Create backup directory
            backup_dir = Path(__file__).parent.parent / 'backups'
            backup_dir.mkdir(exist_ok=True)
            
            # Save backup
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            backup_file = backup_dir / f'prompt_sections_backup_{timestamp}.json'
            
            with open(backup_file, 'w', encoding='utf-8') as f:
                json.dump(sections, f, ensure_ascii=False, indent=2)
            
            print(f"   ✅ Backup saved: {backup_file.name}")
            print(f"   📁 Location: {backup_file}")
        else:
            print("   ⚠️  Table is empty")
            
    except Exception as e:
        print(f"   ❌ Error creating backup: {str(e)}")
        print("   Aborting...")
        return
    
    # Step 2: Confirm deletion
    print("\n" + "="*70)
    print("⚠️  WARNING: This will permanently delete the prompt_sections table!")
    print("="*70)
    print("\n📝 Reason:")
    print("   All prompts are now in app/prompts/*.py files (server-side)")
    print("   The database table is no longer used and can cause confusion")
    print("\n✅ Benefits of removal:")
    print("   • Cleaner database")
    print("   • No confusion about which prompts are used")
    print("   • Prevents accidental use of old system")
    print("   • Single source of truth (code files)")
    
    print("\n❓ Type 'DELETE' (in capital letters) to confirm:")
    confirmation = input("   > ").strip()
    
    if confirmation != 'DELETE':
        print("\n❌ Cancelled. No changes made.")
        print("="*70 + "\n")
        return
    
    # Step 3: Delete the table
    print("\n🗑️  Step 2: Deleting table...")
    print("\n   ⚠️  Note: Supabase Python client cannot drop tables directly.")
    print("   You need to run this SQL command manually in Supabase SQL Editor:\n")
    
    sql_command = "DROP TABLE IF EXISTS prompt_sections CASCADE;"
    
    print("   " + "-"*66)
    print(f"   {sql_command}")
    print("   " + "-"*66)
    
    print("\n   📍 Steps to delete:")
    print("   1. Go to: https://supabase.com/dashboard")
    print("   2. Select your project")
    print("   3. Go to: SQL Editor")
    print("   4. Paste the SQL command above")
    print("   5. Click 'Run'")
    
    print("\n   💡 Or use this direct link:")
    print("   https://supabase.com/dashboard/project/_/sql/new")
    
    # Step 4: Verification instructions
    print("\n✅ Step 3: After deletion, verify:")
    print("   Run this script again to check if table still exists")
    
    print("\n" + "="*70)
    print("📊 SUMMARY")
    print("="*70)
    print(f"\n   ✅ Backup created: {backup_file.name}")
    print("   ⏳ Waiting for manual deletion in Supabase dashboard")
    print("\n   📝 SQL Command:")
    print(f"   {sql_command}")
    
    print("\n🎯 After deletion, your database will only contain:")
    print("   ✅ menu_items - Menu data (dynamic)")
    print("   ✅ orders - Order records")
    print("   ✅ draft_sessions - Active sessions")
    print("   ✅ conversation_history - Chat history")
    print("   ✅ order_items - Order details")
    
    print("\n   ❌ prompt_sections - REMOVED (now in app/prompts/*.py)")
    
    print("\n" + "="*70 + "\n")

if __name__ == "__main__":
    main()

