"""
Script to integrate Structured Outputs into webhook.py
This creates a backup and updates the webhook to use the new service
"""
import os
import sys
import shutil
from pathlib import Path
from datetime import datetime

# Add parent directory to path
sys.path.append(str(Path(__file__).parent.parent))


def create_backup():
    """Create backup of webhook.py"""
    webhook_path = Path(__file__).parent.parent / 'app' / 'routes' / 'webhook.py'
    backup_path = webhook_path.parent / f'webhook.py.backup.{datetime.now().strftime("%Y%m%d_%H%M%S")}'
    
    shutil.copy2(webhook_path, backup_path)
    print(f"✅ Backup created: {backup_path.name}")
    return webhook_path, backup_path


def read_file(path):
    """Read file content"""
    with open(path, 'r', encoding='utf-8') as f:
        return f.read()


def write_file(path, content):
    """Write file content"""
    with open(path, 'w', encoding='utf-8') as f:
        f.write(content)


def integrate_structured_outputs():
    """Integrate structured outputs into webhook.py"""
    print("\n🚀 Integrating Structured Outputs into webhook.py")
    print("="*80)
    
    # Create backup
    print("\n1️⃣ Creating backup...")
    webhook_path, backup_path = create_backup()
    
    # Read current content
    print("\n2️⃣ Reading webhook.py...")
    content = read_file(webhook_path)
    
    # Check if already integrated
    if 'openai_service_structured' in content:
        print("\n⚠️ Structured Outputs already integrated!")
        print("   No changes needed.")
        return
    
    # Make changes
    print("\n3️⃣ Applying changes...")
    
    # Change 1: Update import
    old_import = "from app.services.openai_service import OpenAIService"
    new_import = """from app.services.openai_service_structured import OpenAIServiceStructured
# Legacy import (for fallback)
# from app.services.openai_service import OpenAIService"""
    
    if old_import in content:
        content = content.replace(old_import, new_import)
        print("   ✅ Updated import statement")
    else:
        print("   ⚠️ Could not find import statement")
    
    # Change 2: Update service initialization
    old_init = "openai_service = OpenAIService(credits_manager)"
    new_init = "openai_service = OpenAIServiceStructured(credits_manager)"
    
    if old_init in content:
        content = content.replace(old_init, new_init)
        print("   ✅ Updated service initialization")
    else:
        print("   ⚠️ Could not find service initialization")
    
    # Change 3: Update process_message call
    # This is more complex as it might be in multiple places
    old_call = "await openai_service.process_message("
    new_call = "await openai_service.process_message_structured("
    
    if old_call in content:
        content = content.replace(old_call, new_call)
        print("   ✅ Updated process_message calls")
    else:
        print("   ⚠️ Could not find process_message calls")
    
    # Write updated content
    print("\n4️⃣ Writing updated webhook.py...")
    write_file(webhook_path, content)
    
    print("\n" + "="*80)
    print("✅ Integration complete!")
    print("\n📝 Changes made:")
    print("   1. Updated import to use OpenAIServiceStructured")
    print("   2. Updated service initialization")
    print("   3. Updated process_message calls to process_message_structured")
    print(f"\n💾 Backup saved: {backup_path.name}")
    print("\n🧪 Next steps:")
    print("   1. Review the changes in webhook.py")
    print("   2. Restart the server")
    print("   3. Test with real WhatsApp messages")
    print("   4. Monitor logs for any issues")
    print("\n💡 To rollback:")
    print(f"   cp {backup_path} {webhook_path}")


def show_manual_instructions():
    """Show manual integration instructions"""
    print("\n" + "="*80)
    print("📖 MANUAL INTEGRATION INSTRUCTIONS")
    print("="*80)
    
    print("\nIf automatic integration fails, follow these steps:")
    
    print("\n1️⃣ Update import (around line 20):")
    print("   OLD:")
    print("   from app.services.openai_service import OpenAIService")
    print("\n   NEW:")
    print("   from app.services.openai_service_structured import OpenAIServiceStructured")
    
    print("\n2️⃣ Update service initialization (around line 50):")
    print("   OLD:")
    print("   openai_service = OpenAIService(credits_manager)")
    print("\n   NEW:")
    print("   openai_service = OpenAIServiceStructured(credits_manager)")
    
    print("\n3️⃣ Update process_message calls (around line 425):")
    print("   OLD:")
    print("   ai_response = await openai_service.process_message(...)")
    print("\n   NEW:")
    print("   ai_response = await openai_service.process_message_structured(...)")
    
    print("\n4️⃣ Restart server:")
    print("   pkill -f uvicorn")
    print("   cd restaurant-chatbot-server")
    print("   source venv/bin/activate")
    print("   python -m uvicorn app.main:app --host 0.0.0.0 --port 8000 --reload")


def main():
    """Main function"""
    print("\n🔧 Structured Outputs Integration Tool")
    print("="*80)
    
    try:
        integrate_structured_outputs()
        show_manual_instructions()
        
        print("\n" + "="*80)
        print("✅ Done!")
        print("\n💡 Remember to:")
        print("   1. Review the changes")
        print("   2. Restart the server")
        print("   3. Test thoroughly")
        
    except Exception as e:
        print(f"\n❌ Error: {e}")
        import traceback
        traceback.print_exc()
        print("\n💡 Please follow the manual instructions above")
        return 1
    
    return 0


if __name__ == '__main__':
    sys.exit(main())

