#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
إعادة بناء فهرس FAISS مع البيانات المحسّنة
يستخدم aliases وsearch_keywords لتحسين البحث الدلالي

تشغيل من مجلد المشروع:
cd restaurant-chatbot-server
python3 -m scripts.improvement.rebuild_faiss_index
"""

import json
import asyncio
import sys

try:
    from app.services.faiss_menu_search import faiss_menu_search
except ImportError:
    print("❌ خطأ: لا يمكن استيراد الموديولات")
    print("   قم بتشغيل السكريبت من مجلد المشروع:")
    print("   cd restaurant-chatbot-server")
    print("   python3 -m scripts.improvement.rebuild_faiss_index")
    sys.exit(1)


async def rebuild_index():
    """إعادة بناء فهرس FAISS"""
    
    print("🚀 إعادة بناء فهرس FAISS")
    print("="*70)
    
    # 1. قراءة المنيو المحسّن
    menu_file = 'data/menu_items_local.json'
    
    try:
        with open(menu_file, 'r', encoding='utf-8') as f:
            menu_items = json.load(f)
    except FileNotFoundError:
        print(f"❌ الملف غير موجود: {menu_file}")
        return False
    
    print(f"📊 عدد الأصناف: {len(menu_items)}")
    
    # 2. التحقق من وجود aliases وkeywords
    items_with_aliases = sum(1 for item in menu_items if item.get('aliases'))
    items_with_keywords = sum(1 for item in menu_items if item.get('search_keywords'))
    
    print(f"   أصناف مع aliases: {items_with_aliases}/{len(menu_items)}")
    print(f"   أصناف مع keywords: {items_with_keywords}/{len(menu_items)}")
    
    if items_with_aliases == 0 or items_with_keywords == 0:
        print("\n⚠️ تحذير: بعض الأصناف لا تحتوي على aliases أو keywords")
        print("   قم بتشغيل: python3 scripts/improvement/enhance_menu_data.py")
        return False
    
    # 3. إعادة بناء الفهرس
    print(f"\n🔄 إعادة بناء الفهرس...")
    
    try:
        await faiss_menu_search.load_menu(menu_items)
        print(f"✅ تم بناء الفهرس بنجاح!")
    except Exception as e:
        print(f"❌ خطأ في بناء الفهرس: {e}")
        return False
    
    # 4. عرض إحصائيات الفهرس
    stats = faiss_menu_search.get_stats()
    
    print(f"\n📊 إحصائيات الفهرس:")
    print(f"   عدد الأصناف: {stats['items_count']}")
    print(f"   البُعد: {stats['dimension']}")
    print(f"   الذاكرة: {stats['memory_mb']:.1f} MB")
    print(f"   النموذج: {stats['model']}")
    
    # 5. اختبار البحث
    print(f"\n🔍 اختبار البحث:")
    
    test_queries = [
        "حموص",
        "كبه",
        "بابا غنوج",
        "سمبوسك",
        "يبرق",
        "هريسة",
        "برغر",
        "مويه",
    ]
    
    for query in test_queries:
        results = await faiss_menu_search.search(
            query=query,
            top_k=3,
            min_score=0.3,
            keyword_filter=True,
            exact_match_boost=True
        )
        
        if results:
            top_result = results[0]
            print(f"   ✅ '{query}' → {top_result[0]['name_ar']} ({top_result[1]:.1f}%)")
        else:
            print(f"   ❌ '{query}' → لا توجد نتائج")
    
    print(f"\n✅ اكتمل إعادة البناء!")
    return True


if __name__ == '__main__':
    success = asyncio.run(rebuild_index())
    sys.exit(0 if success else 1)

