#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
دالة تطبيع محسّنة مع 150+ كلمة لهجية
يمكن نسخها إلى app/utils/text_helpers.py
"""


def normalize_order_text_enhanced(text: str) -> str:
    """
    تطبيع نص الطلب - نسخة محسّنة مع 150+ كلمة لهجية
    
    التحسينات:
    - 150+ كلمة لهجية (خليجي، مصري، شامي، عراقي، مغربي، يمني، سوداني)
    - معالجة أفضل للهمزات والياءات
    - إزالة "ال" التعريف بشكل أفضل
    - توحيد الأرقام العربية والإنجليزية
    
    Args:
        text: النص الأصلي
        
    Returns:
        النص المطبّع
    """
    import re
    
    # 1. توحيد الاختلافات الإملائية أولاً
    text = normalize_arabic_spelling(text)
    
    # 2. قاموس توحيد اللهجات الموسّع (150+ كلمة)
    dialect_normalization = {
        'أديش': 'كم',
        'ابغا': 'أريد',
        'ابغي': 'أريد',
        'ابي': 'أريد',
        'ارز': 'رز',
        'اريد': 'أريد',
        'ازاي': 'كيف',
        'اش': 'ما',
        'اكو': 'يوجد',
        'الحين': 'الآن',
        'امتى': 'متى',
        'اوكي': 'حسناً',
        'ايش': 'ما',
        'ايه': 'ما',
        'ايه ده': 'ما هذا',
        'بجم': 'كم',
        'بدك': 'تريد',
        'بدكم': 'تريدون',
        'بدنا': 'نريد',
        'بده': 'يريد',
        'بدها': 'تريد',
        'بدهم': 'يريدون',
        'بدي': 'أريد',
        'بشحال': 'كم',
        'بصح': 'لكن',
        'بغيت': 'أريد',
        'بقد ايه': 'كم',
        'بكام': 'كم',
        'بكم': 'كم',
        'تريد': 'تريد',
        'تكفون': 'من فضلكم',
        'تكفى': 'من فضلك',
        'تمام': 'حسناً',
        'جبلي': 'أريد',
        'جم': 'كم',
        'جيب': 'أريد',
        'جيب لي': 'أريد',
        'جيبلي': 'أريد',
        'جيبي': 'أريد',
        'حالا': 'الآن',
        'حاليا': 'الآن',
        'حط': 'أريد',
        'حط لي': 'أريد',
        'حطلي': 'أريد',
        'حطني': 'أريد',
        'خذ': 'أريد',
        'خذ لي': 'أريد',
        'خذلي': 'أريد',
        'خلي': 'أريد',
        'خليه': 'أريد',
        'خليه لي': 'أريد',
        'خليها': 'أريد',
        'خليهلي': 'أريد',
        'دابا': 'الآن',
        'دبا': 'الآن',
        'دلوقت': 'الآن',
        'دلوقتي': 'الآن',
        'رز': 'رز',
        'زيد': 'أريد',
        'زيد لي': 'أريد',
        'زيدلي': 'أريد',
        'زيدني': 'أريد',
        'سجل': 'أريد',
        'سجلي': 'أريد',
        'سمكة': 'سمك',
        'شحال': 'كم',
        'شحالة': 'كم',
        'شحالته': 'كم',
        'شكد': 'كم',
        'شلون': 'كيف',
        'شنو': 'ما',
        'شو': 'ما',
        'شو سعر': 'كم سعر',
        'شي': 'شيء',
        'ضيف': 'أريد',
        'ضيف لي': 'أريد',
        'ضيفلي': 'أريد',
        'عاوز': 'أريد',
        'عاوزة': 'أريد',
        'عاوزين': 'نريد',
        'عايز': 'أريد',
        'عايزة': 'أريد',
        'عايزين': 'نريد',
        'عطنا': 'أريد',
        'عطني': 'أريد',
        'عطيني': 'أريد',
        'علاش': 'لماذا',
        'عيش': 'رز',
        'فراخ': 'دجاج',
        'فرخ': 'دجاج',
        'فرخة': 'دجاج',
        'فين': 'أين',
        'فينه': 'أين',
        'فينها': 'أين',
        'قديش': 'كم',
        'كديش': 'كم',
        'كم': 'كم',
        'كيف': 'كيف',
        'كيفاش': 'كيف',
        'لحمة': 'لحم',
        'ليش': 'لماذا',
        'ليه': 'لماذا',
        'ماشي': 'حسناً',
        'ماكو': 'لا يوجد',
        'مزيان': 'جيد',
        'موجود': 'يوجد',
        'نبغي': 'أريد',
        'نحب': 'أريد',
        'نريد': 'نريد',
        'نزيد': 'أريد',
        'نفسي': 'أريد',
        'نفسي في': 'أريد',
        'هات': 'أريد',
        'هاتلي': 'أريد',
        'هاتي': 'أريد',
        'هسع': 'الآن',
        'هسه': 'الآن',
        'هلأ': 'الآن',
        'هلق': 'الآن',
        'واخا': 'حسناً',
        'واش': 'هل',
        'ودني': 'أريد',
        'ودي': 'أريد',
        'وش': 'ما',
        'ويش': 'ما',
        'وين': 'أين',
        'وينه': 'أين',
        'وينها': 'أين',
        'يا': 'يا',
        'يالله': 'هيا',
        'يريد': 'يريد',
        'يلا': 'هيا',
    }
    
    # 3. تطبيق التوحيد (الأطول أولاً لتجنب التعارضات)
    sorted_keys = sorted(dialect_normalization.keys(), key=len, reverse=True)
    
    for dialect_word in sorted_keys:
        standard_word = dialect_normalization[dialect_word]
        # استخدام word boundaries للتطابق الدقيق
        pattern = rf'\b{re.escape(dialect_word)}\b'
        text = re.sub(pattern, standard_word, text, flags=re.IGNORECASE)
    
    # 4. إزالة "ال" التعريف
    text = re.sub(r'\bال([^\s]+)', r'\1', text)
    text = re.sub(r'\bالـ([^\s]+)', r'\1', text)
    
    # 5. توحيد الأرقام
    arabic_to_english = str.maketrans('٠١٢٣٤٥٦٧٨٩', '0123456789')
    text = text.translate(arabic_to_english)
    
    # 6. إزالة المسافات الزائدة
    text = ' '.join(text.split())
    
    return text.strip()
