#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
تحسين بيانات المنيو
إضافة aliases، search_keywords، وتحسين tags
"""

import json
from pathlib import Path
from typing import List, Dict


# قاموس الأسماء البديلة (aliases) لكل صنف
# يتم المطابقة بناءً على الكلمات المفتاحية في الاسم
MENU_ALIASES = {
    # === مقبلات ===
    "حمص": ["حمص", "حموص", "حمص بالطحينة", "حمص شامي"],
    "كبة": ["كبة", "كبة مقلية", "كبة لبنانية", "كبه", "كبه مقلية"],
    "متبل": ["متبل", "بابا غنوج", "باذنجان مشوي", "متبل باذنجان"],
    "سمبوسة": ["سمبوسة", "سمبوسك", "سمبوسة لحم", "سمبوسه"],
    "ورق عنب": ["ورق عنب", "ورق العنب", "محشي ورق عنب", "يبرق"],
    "فتوش": ["فتوش", "فتوشة", "سلطة فتوش"],
    "تبولة": ["تبولة", "تبوله", "سلطة تبولة"],
    "سلطة": ["سلطة", "سلطه", "سلطات"],

    # === أطباق عمانية ===
    "عرسي": ["عرسي", "عرسي جعلان", "عرسي غنم", "عرسي لحم"],
    "هريس": ["هريس", "هريسة", "حريس", "هريس دجاج", "هريس لحم"],
    "مصلونة": ["مصلونة", "مصلونه", "مصلونة دجاج", "مصلونة لحم", "مصلونة سمك"],
    "مصورة": ["مصورة", "مصوره", "مصورة أول"],
    "قاشع": ["قاشع", "قاشع برية", "قاشع بري"],
    "صحنة": ["صحنة", "صحنه"],
    "مالح": ["مالح", "مالح رز"],

    # === لحوم ===
    "لحم جمل": ["لحم جمل", "جمل", "لحم الجمل", "قصابية"],
    "كبسة": ["كبسة", "كبسه", "كبسة عوان"],

    # === أسماك ===
    "سمك": ["سمك", "سمكة", "سمج", "سمك مقلي", "سمك تونة", "تونة"],

    # === مشروبات ===
    "عصير": ["عصير", "عصير طازج", "عصير فريش", "عصاير"],
    "ماء": ["ماء", "مياه", "ماي", "مويه"],
    "مشروب غازي": ["مشروب غازي", "مشروبات غازية", "بيبسي", "كولا", "سفن اب", "غازي", "كوكاكولا"],
    "لبن": ["لبن", "لبن رايب", "روب"],

    # === برجر ===
    "برجر": ["برجر", "برغر", "برقر", "همبرجر", "برجر كلاسيك"],

    # === بطاطس ===
    "بطاطس": ["بطاطس", "بطاطا", "فرايز", "بطاطس مقلية"],

    # === شوربات ===
    "شوربة": ["شوربة", "شوربه", "حساء"],
    "شوربة عدس": ["شوربة عدس", "عدس", "حساء عدس"],
    "شوربة ذرة": ["شوربة ذرة", "ذرة", "حساء ذرة"],
    "شوربة مشروم": ["شوربة مشروم", "مشروم", "فطر", "حساء مشروم"],
    "شوربة هريس": ["شوربة هريس", "هريس"],
    "شوربة روبيان": ["شوربة روبيان", "روبيان", "جمبري", "بايبلو"],

    # === عروض ===
    "عرض": ["عرض", "وجبة", "وجبة عائلية", "عرض خاص"],
}


# كلمات البحث الإضافية (search_keywords)
SEARCH_KEYWORDS = {
    # === مقبلات ===
    "حمص": ["حمص", "حموص", "طحينة", "مقبلات", "مزة"],
    "كبة مقلية": ["كبة", "كبه", "برغل", "لحم", "مقلي", "مقرمش"],
    "متبل": ["متبل", "بابا غنوج", "باذنجان", "طحينة"],
    "سمبوسة باللحم": ["سمبوسة", "سمبوسك", "لحم", "محشي", "مقرمش"],
    "ورق عنب": ["ورق عنب", "يبرق", "محشي", "أرز", "لحم"],
    
    # === أطباق رئيسية ===
    "كبسة": ["كبسة", "رز", "دجاج", "لحم", "غنم", "بهارات"],
    "مندي": ["مندي", "رز", "دجاج", "لحم", "تنور", "يمني"],
    "مظبي": ["مظبي", "رز", "دجاج", "لحم", "مشوي"],
    "برياني": ["برياني", "بريانى", "رز", "دجاج", "لحم", "هندي"],
    "مقلقل": ["مقلقل", "لحم", "طماطم", "بصل"],
    "هريس": ["هريس", "حريس", "قمح", "دجاج", "لحم", "عماني"],
    
    # === دجاج ===
    "دجاج مشوي": ["دجاج", "فراخ", "مشوي", "فحم", "شواية"],
    "دجاج مقلي": ["دجاج", "فراخ", "مقلي", "محمر", "كرسبي"],
    "شاورما دجاج": ["شاورما", "دجاج", "ساندويتش", "لفة"],
    
    # === لحوم ===
    "لحم مشوي": ["لحم", "لحمة", "مشوي", "فحم", "غنم"],
    "كباب": ["كباب", "كبب", "لحم", "مشوي", "سيخ"],
    "ريش": ["ريش", "ريوش", "غنم", "مشوي", "فحم"],
    
    # === أسماك ===
    "سمك مشوي": ["سمك", "سمكة", "سمج", "مشوي", "بحري"],
    "سمك مقلي": ["سمك", "سمكة", "مقلي", "محمر"],
    "روبيان": ["روبيان", "جمبري", "قريدس", "بحري"],
    
    # === مشروبات ===
    "عصير طازج": ["عصير", "فريش", "طازج", "برتقال", "تفاح", "مانجو"],
    "ماء": ["ماء", "مياه", "ماي", "مويه"],
    "شاي": ["شاي", "شاهي", "تشاي", "ساخن"],
    "قهوة": ["قهوة", "كوفي", "عربية", "عمانية"],
}


def enhance_menu_item(item: Dict) -> Dict:
    """تحسين صنف واحد من المنيو"""

    name_ar = item.get('name_ar', '')

    # البحث عن aliases بناءً على الكلمات المفتاحية
    aliases = [name_ar]  # نبدأ بالاسم الأساسي

    for key, values in MENU_ALIASES.items():
        # مطابقة أفضل: نبحث عن الكلمة المفتاحية في الاسم
        if key in name_ar:
            # نضيف جميع aliases ما عدا المكررات
            for alias in values:
                if alias not in aliases:
                    aliases.append(alias)

    # إذا لم نجد أي aliases، نضيف variations بسيطة
    if len(aliases) == 1:
        # إضافة نسخة بدون "ال"
        if name_ar.startswith('ال'):
            aliases.append(name_ar[2:])
        # إضافة نسخة مع تاء مربوطة/هاء
        if name_ar.endswith('ة'):
            aliases.append(name_ar[:-1] + 'ه')
        elif name_ar.endswith('ه'):
            aliases.append(name_ar[:-1] + 'ة')

    # البحث عن search_keywords
    search_keywords = [name_ar]  # نبدأ بالاسم الأساسي

    for key, values in SEARCH_KEYWORDS.items():
        if key in name_ar:
            for keyword in values:
                if keyword not in search_keywords:
                    search_keywords.append(keyword)

    # إضافة tags كـ keywords
    for tag in item.get('tags', []):
        if tag not in search_keywords:
            search_keywords.append(tag)

    # إضافة كلمات من الاسم
    words = name_ar.split()
    for word in words:
        if word not in search_keywords and len(word) > 2:
            search_keywords.append(word)

    # إضافة الحقول الجديدة
    item['aliases'] = aliases
    item['search_keywords'] = search_keywords

    return item


def enhance_menu_file(input_file: str, output_file: str):
    """تحسين ملف المنيو بالكامل"""
    
    print("🚀 تحسين بيانات المنيو")
    print("="*70)
    
    # قراءة الملف
    with open(input_file, 'r', encoding='utf-8') as f:
        menu_items = json.load(f)
    
    print(f"📊 عدد الأصناف: {len(menu_items)}")
    
    # تحسين كل صنف
    enhanced_items = []
    stats = {
        'total': len(menu_items),
        'with_aliases': 0,
        'with_keywords': 0,
        'total_aliases': 0,
        'total_keywords': 0,
    }
    
    for item in menu_items:
        enhanced = enhance_menu_item(item)
        enhanced_items.append(enhanced)
        
        # إحصائيات
        if enhanced.get('aliases'):
            stats['with_aliases'] += 1
            stats['total_aliases'] += len(enhanced['aliases'])
        
        if enhanced.get('search_keywords'):
            stats['with_keywords'] += 1
            stats['total_keywords'] += len(enhanced['search_keywords'])
    
    # حفظ الملف المحسّن
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(enhanced_items, f, ensure_ascii=False, indent=2)
    
    print(f"\n✅ تم حفظ الملف المحسّن: {output_file}")
    
    # عرض الإحصائيات
    print(f"\n📊 الإحصائيات:")
    print(f"   إجمالي الأصناف:        {stats['total']}")
    print(f"   أصناف مع aliases:      {stats['with_aliases']} ({stats['with_aliases']/stats['total']*100:.1f}%)")
    print(f"   أصناف مع keywords:     {stats['with_keywords']} ({stats['with_keywords']/stats['total']*100:.1f}%)")
    print(f"   إجمالي aliases:        {stats['total_aliases']}")
    print(f"   إجمالي keywords:       {stats['total_keywords']}")
    print(f"   متوسط aliases/صنف:     {stats['total_aliases']/stats['total']:.1f}")
    print(f"   متوسط keywords/صنف:    {stats['total_keywords']/stats['total']:.1f}")
    
    # عرض أمثلة
    print(f"\n💡 أمثلة على التحسينات:")
    for i, item in enumerate(enhanced_items[:5]):
        print(f"\n   {i+1}. {item['name_ar']}")
        print(f"      aliases: {', '.join(item.get('aliases', [])[:3])}")
        print(f"      keywords: {', '.join(item.get('search_keywords', [])[:5])}")


if __name__ == '__main__':
    input_file = 'data/menu_items_local.json'
    output_file = 'data/menu_items_enhanced.json'
    
    enhance_menu_file(input_file, output_file)
    
    print("\n✅ اكتمل التحسين!")
    print("\n📝 الخطوات التالية:")
    print("   1. راجع الملف: data/menu_items_enhanced.json")
    print("   2. اختبر البحث مع الأسماء البديلة")
    print("   3. إذا كان كل شيء جيد، استبدل menu_items_local.json")

