#!/usr/bin/env python3
"""
Generate embeddings for menu items and store them in Supabase
This allows for semantic search of menu items using vector similarity
"""

import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from openai import OpenAI
from supabase import create_client
from dotenv import load_dotenv
import logging

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s | %(levelname)-8s | %(message)s',
    datefmt='%Y-%m-%d %H:%M:%S'
)
logger = logging.getLogger(__name__)

# Load environment variables
load_dotenv()

# Initialize clients
openai_client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))
supabase = create_client(
    os.getenv("SUPABASE_URL"),
    os.getenv("SUPABASE_SERVICE_KEY")
)


def get_embedding(text: str) -> list[float]:
    """
    Get embedding vector for text using OpenAI's text-embedding-3-small model
    
    Args:
        text: Text to embed
        
    Returns:
        List of floats representing the embedding vector
    """
    try:
        response = openai_client.embeddings.create(
            model="text-embedding-3-small",
            input=text,
            dimensions=1536  # Using 1536 dimensions for compatibility
        )
        return response.data[0].embedding
    except Exception as e:
        logger.error(f"Error getting embedding: {e}")
        raise


def generate_menu_embeddings():
    """
    Generate embeddings for all active menu items and store in database
    """
    try:
        # 1. Fetch all active menu items
        logger.info("📥 Fetching menu items from database...")
        response = supabase.table("menu_items").select("*").eq("active", True).execute()
        menu_items = response.data
        logger.info(f"✅ Found {len(menu_items)} active menu items")
        
        # 2. Clear existing embeddings
        logger.info("🗑️  Clearing existing embeddings...")
        supabase.table("menu_embeddings").delete().neq("id", 0).execute()
        logger.info("✅ Cleared existing embeddings")
        
        # 3. Generate embeddings for each item
        logger.info("🔨 Generating embeddings...")
        embeddings_data = []
        
        for idx, item in enumerate(menu_items, 1):
            code = item['code']
            name_ar = item['name_ar']
            name_en = item['name_en']
            
            # Create search text (both Arabic and English for better matching)
            search_text = f"{name_ar} {name_en}"
            
            logger.info(f"  [{idx}/{len(menu_items)}] Processing: {name_ar} / {name_en}")
            
            # Get embedding
            embedding = get_embedding(search_text)
            
            # Prepare data
            embeddings_data.append({
                'menu_item_code': code,
                'name_ar': name_ar,
                'name_en': name_en,
                'search_text': search_text,
                'embedding': embedding
            })
        
        # 4. Insert embeddings in batch
        logger.info("💾 Inserting embeddings into database...")
        supabase.table("menu_embeddings").insert(embeddings_data).execute()
        logger.info(f"✅ Successfully inserted {len(embeddings_data)} embeddings")
        
        # 5. Test search
        logger.info("\n🧪 Testing search with 'حمص'...")
        test_embedding = get_embedding("حمص")
        
        # Call the search function
        result = supabase.rpc(
            'search_menu_items',
            {
                'query_embedding': test_embedding,
                'match_count': 5
            }
        ).execute()
        
        logger.info("📊 Top 5 matches:")
        for match in result.data:
            logger.info(f"  - {match['name_ar']} / {match['name_en']} (similarity: {match['similarity']:.3f})")
        
        logger.info("\n✅ All done! Menu embeddings generated successfully!")
        
    except Exception as e:
        logger.error(f"❌ Error: {e}")
        raise


if __name__ == "__main__":
    generate_menu_embeddings()

