#!/usr/bin/env python3
"""
Expression Generator - مولد العبارات التلقائي
يولد آلاف العبارات من قوالب أساسية
"""

import yaml
import json
import os
import sys
from typing import List, Dict, Set
from itertools import product

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from app.utils.logger import app_logger as logger


class ExpressionGenerator:
    """مولد العبارات التلقائي"""
    
    def __init__(self):
        self.generated_expressions: Dict[str, Dict] = {}
        self.menu_items: List[Dict] = []
        self._load_menu_items()
    
    def _load_menu_items(self):
        """تحميل أصناف المنيو من الكاش"""
        try:
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(__file__)),
                'data', 'cache', 'menu_cache.json'
            )
            
            if os.path.exists(cache_path):
                with open(cache_path, 'r', encoding='utf-8') as f:
                    cache_data = json.load(f)
                    self.menu_items = cache_data.get('items', [])
                    logger.info(f"✅ Loaded {len(self.menu_items)} menu items")
            else:
                logger.warning("⚠️ Menu cache not found")
        except Exception as e:
            logger.error(f"❌ Error loading menu items: {e}")
    
    def generate_new_order_expressions(self) -> List[str]:
        """توليد عبارات طلب جديد"""
        expressions = set()
        
        # 1. أفعال الطلب الأساسية
        base_verbs = {
            'gulf': ['أبغى', 'أبي', 'ابغى', 'ابي', 'بغيت', 'ودي', 'عطني', 'عطنا', 'خذ', 'خذ لي', 'حط', 'حط لي'],
            'egyptian': ['عايز', 'عايزة', 'نفسي في', 'هات', 'هاتي', 'جيب', 'جيبي', 'حطلي'],
            'levantine': ['بدي', 'بدنا', 'بدك', 'جِبلي', 'عطيني', 'زيدلي'],
            'iraqi': ['اريد', 'نريد', 'ريدني', 'جيبلي'],
            'maghrebi': ['نحب', 'نبغي', 'بغيت', 'حطني', 'زيدني'],
            'formal': ['أريد', 'أرغب في', 'أطلب', 'أحتاج', 'أود', 'أحب', 'أفضل']
        }
        
        # 2. الكميات
        quantities = ['', '1', '2', '3', '4', '5', '٢', '٣', '٤', '٥', 
                     'واحد', 'اثنين', 'ثلاثة', 'أربعة', 'خمسة',
                     'طبق', 'طبقين', 'ثلاث أطباق']
        
        # 3. حروف الجر
        prepositions = ['', 'لي', 'لنا', 'معاي', 'معانا']
        
        # 4. توليد العبارات الأساسية (بدون أصناف)
        for dialect, verbs in base_verbs.items():
            for verb in verbs:
                for qty in quantities[:5]:  # أول 5 كميات فقط
                    for prep in prepositions[:3]:  # أول 3 حروف جر
                        if qty and prep:
                            expr = f"{verb} {qty} {prep}".strip()
                        elif qty:
                            expr = f"{verb} {qty}".strip()
                        elif prep:
                            expr = f"{verb} {prep}".strip()
                        else:
                            expr = verb.strip()
                        
                        expressions.add(expr)
        
        # 5. توليد عبارات مع أصناف المنيو
        for item in self.menu_items[:50]:  # أول 50 صنف
            name_ar = item.get('name_ar', '').strip()
            if not name_ar:
                continue
            
            # مع كل فعل
            for dialect, verbs in base_verbs.items():
                for verb in verbs[:3]:  # أول 3 أفعال من كل لهجة
                    # بدون كمية
                    expressions.add(f"{verb} {name_ar}")
                    
                    # مع كمية
                    for qty in ['2', '3', '٢', '٣', 'اثنين', 'ثلاثة']:
                        expressions.add(f"{verb} {qty} {name_ar}")
        
        logger.info(f"✅ Generated {len(expressions)} new_order expressions")
        return list(expressions)
    
    def generate_quantity_expressions(self) -> Dict[str, List[str]]:
        """توليد عبارات الكمية"""
        expressions = {
            'exact': set(),
            'increase': set(),
            'decrease': set()
        }
        
        # 1. كميات محددة
        numbers = ['1', '2', '3', '4', '5', '٢', '٣', '٤', '٥',
                  'واحد', 'اثنين', 'ثلاثة', 'أربعة', 'خمسة']
        
        exact_templates = [
            'خليها {num}',
            'اجعلها {num}',
            'غيرها إلى {num}',
            'بدلها بـ {num}',
            'أريد {num}',
            'عايز {num}',
            'أبغى {num}',
            '{num} بس',
            '{num} فقط'
        ]
        
        for template in exact_templates:
            for num in numbers:
                expressions['exact'].add(template.replace('{num}', num))
        
        # 2. زيادة الكمية
        increase_verbs = [
            'زيد', 'زود', 'ضيف', 'أضف', 'زيادة', 'إضافة',
            'ضاعف', 'ضاعفها', 'زيد الكمية', 'زود الكمية',
            'أكثر', 'المزيد', 'كمان', 'زيادة واحد'
        ]
        
        for verb in increase_verbs:
            expressions['increase'].add(verb)
            expressions['increase'].add(f"{verb} لي")
            expressions['increase'].add(f"{verb} واحد")
            expressions['increase'].add(f"{verb} اثنين")
        
        # 3. تقليل الكمية
        decrease_verbs = [
            'نقص', 'قلل', 'أنقص', 'خفف', 'تقليل',
            'نقص الكمية', 'قلل الكمية', 'أقل',
            'نقص واحد', 'احذف واحد', 'شيل واحد'
        ]
        
        for verb in decrease_verbs:
            expressions['decrease'].add(verb)
            expressions['decrease'].add(f"{verb} لي")
            expressions['decrease'].add(f"{verb} واحد")
            expressions['decrease'].add(f"{verb} اثنين")
        
        result = {k: list(v) for k, v in expressions.items()}
        total = sum(len(v) for v in result.values())
        logger.info(f"✅ Generated {total} quantity expressions")
        return result
    
    def generate_confirmation_expressions(self) -> Dict[str, List[str]]:
        """توليد عبارات التأكيد"""
        expressions = {
            'accept': set(),
            'reject': set()
        }
        
        # 1. قبول
        accept_base = [
            'نعم', 'yes', 'أيوه', 'ايوه', 'اه', 'آه', 'أكيد', 'اكيد',
            'تمام', 'تماام', 'تمااام', 'ok', 'okay', 'اوكي', 'اوكيه',
            'صح', 'صحيح', 'بالضبط', 'زين', 'ماشي', 'يب', 'yep', 'yeah',
            'موافق', 'موافقة', 'أوافق', 'اوافق', 'طيب', 'حلو'
        ]
        
        for word in accept_base:
            expressions['accept'].add(word)
            expressions['accept'].add(f"{word} تمام")
            expressions['accept'].add(f"{word} أكيد")
            expressions['accept'].add(f"{word} موافق")
        
        # 2. رفض
        reject_base = [
            'لا', 'no', 'لأ', 'لاء', 'مو', 'ما', 'not', 'nope',
            'مش', 'مو عايز', 'ما أبغى', 'ما أبي', 'مش عايز',
            'لا شكراً', 'لا شكرا', 'خلاص', 'كفاية', 'بس', 'enough',
            'مو لازم', 'ما أحتاج', 'غير مهتم', 'مش محتاج'
        ]
        
        for word in reject_base:
            expressions['reject'].add(word)
            expressions['reject'].add(f"{word} شكراً")
            expressions['reject'].add(f"{word} خلاص")
        
        result = {k: list(v) for k, v in expressions.items()}
        total = sum(len(v) for v in result.values())
        logger.info(f"✅ Generated {total} confirmation expressions")
        return result
    
    def generate_inquiry_expressions(self) -> List[str]:
        """توليد عبارات الاستفسار"""
        expressions = set()
        
        # 1. استفسار عن السعر
        price_templates = [
            'كم سعر', 'بكم', 'كم ثمن', 'كم قيمة', 'كم تكلفة',
            'بكام', 'بكم سعر', 'شحالة', 'وش سعر', 'ايش سعر',
            'how much', 'price', 'cost'
        ]
        
        for template in price_templates:
            expressions.add(template)
            expressions.add(f"{template} هذا")
            expressions.add(f"{template} هذي")
        
        # مع أصناف المنيو
        for item in self.menu_items[:20]:  # أول 20 صنف
            name_ar = item.get('name_ar', '').strip()
            if name_ar:
                for template in price_templates[:5]:
                    expressions.add(f"{template} {name_ar}")
        
        # 2. استفسار عن التوفر
        availability_templates = [
            'عندكم', 'عندك', 'فيه', 'في', 'موجود', 'متوفر',
            'يوجد', 'هل يوجد', 'هل عندكم', 'هل فيه',
            'do you have', 'available', 'have'
        ]
        
        for template in availability_templates:
            expressions.add(template)
        
        # مع أصناف المنيو
        for item in self.menu_items[:20]:
            name_ar = item.get('name_ar', '').strip()
            if name_ar:
                for template in availability_templates[:5]:
                    expressions.add(f"{template} {name_ar}")
                    expressions.add(f"{template} {name_ar}؟")
        
        # 3. استفسار عن المنيو
        menu_templates = [
            'المنيو', 'القائمة', 'menu', 'عندكم ايش', 'عندكم شنو',
            'وش عندكم', 'ايش عندكم', 'شو عندكم', 'what do you have',
            'عرض المنيو', 'اعرض القائمة', 'show menu'
        ]
        
        expressions.update(menu_templates)
        
        logger.info(f"✅ Generated {len(expressions)} inquiry expressions")
        return list(expressions)
    
    def generate_all(self) -> Dict[str, any]:
        """توليد جميع العبارات"""
        logger.info("🚀 Starting expression generation...")
        
        result = {
            'new_order': {
                'all_dialects': self.generate_new_order_expressions()
            },
            'quantity': self.generate_quantity_expressions(),
            'confirmation': self.generate_confirmation_expressions(),
            'inquiry': self.generate_inquiry_expressions()
        }
        
        # حساب الإجمالي
        total = 0
        total += len(result['new_order']['all_dialects'])
        total += sum(len(v) for v in result['quantity'].values())
        total += sum(len(v) for v in result['confirmation'].values())
        total += len(result['inquiry'])
        
        logger.info(f"✅ Total expressions generated: {total}")
        
        return result
    
    def save_to_yaml(self, output_path: str):
        """حفظ العبارات في ملف YAML"""
        expressions = self.generate_all()
        
        with open(output_path, 'w', encoding='utf-8') as f:
            yaml.dump(expressions, f, allow_unicode=True, sort_keys=False)
        
        logger.info(f"✅ Saved to {output_path}")
    
    def save_to_json(self, output_path: str):
        """حفظ العبارات في ملف JSON"""
        expressions = self.generate_all()
        
        with open(output_path, 'w', encoding='utf-8') as f:
            json.dump(expressions, f, ensure_ascii=False, indent=2)
        
        logger.info(f"✅ Saved to {output_path}")


def main():
    """الدالة الرئيسية"""
    print("="*70)
    print("🤖 Expression Generator - مولد العبارات التلقائي")
    print("="*70)
    
    generator = ExpressionGenerator()
    
    # توليد العبارات
    expressions = generator.generate_all()
    
    # عرض الإحصائيات
    print("\n📊 Statistics:")
    print("-"*70)
    print(f"new_order: {len(expressions['new_order']['all_dialects'])} expressions")
    print(f"quantity: {sum(len(v) for v in expressions['quantity'].values())} expressions")
    print(f"  - exact: {len(expressions['quantity']['exact'])}")
    print(f"  - increase: {len(expressions['quantity']['increase'])}")
    print(f"  - decrease: {len(expressions['quantity']['decrease'])}")
    print(f"confirmation: {sum(len(v) for v in expressions['confirmation'].values())} expressions")
    print(f"  - accept: {len(expressions['confirmation']['accept'])}")
    print(f"  - reject: {len(expressions['confirmation']['reject'])}")
    print(f"inquiry: {len(expressions['inquiry'])} expressions")
    
    total = (len(expressions['new_order']['all_dialects']) +
             sum(len(v) for v in expressions['quantity'].values()) +
             sum(len(v) for v in expressions['confirmation'].values()) +
             len(expressions['inquiry']))
    
    print(f"\n✅ Total: {total} expressions")
    print("-"*70)
    
    # حفظ النتائج
    output_dir = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'data', 'generated')
    os.makedirs(output_dir, exist_ok=True)
    
    yaml_path = os.path.join(output_dir, 'generated_expressions.yaml')
    json_path = os.path.join(output_dir, 'generated_expressions.json')
    
    generator.save_to_yaml(yaml_path)
    generator.save_to_json(json_path)
    
    print(f"\n✅ Files saved:")
    print(f"   - {yaml_path}")
    print(f"   - {json_path}")
    print("\n" + "="*70)


if __name__ == '__main__':
    main()

