#!/usr/bin/env python3
"""
Fix 'Yes' Intent Detection
The AI is incorrectly detecting "نعم" as Continue Order instead of adding more items
"""
import os
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))

print("="*70)
print("🔧 Fixing 'نعم' intent detection")
print("="*70)

# The problem: AI thinks "نعم" after adding items means "continue" (which empties the session)
# Solution: Make it clear that "نعم" means "add more items" (Ask Information intent)

intent_continue_fixed = """
**Intent: Continue Order (متابعة الطلب)**

**⚠️ CRITICAL: This intent should ONLY be used when customer wants to PROCEED to next stage!**

**متى تستخدم هذه النية:**
- عندما يقول العميل **صراحة** أنه يريد المتابعة: "تمام"، "خلاص"، "ماشي"، "لا" (بعد سؤال إضافة المزيد)
- عندما يقول "أرسل الموقع"، "send location"

**⚠️ لا تستخدم هذه النية إذا:**
- العميل يقول "نعم" أو "yes" → استخدم **Ask Information** (يريد إضافة المزيد!)
- العميل يطلب صنف جديد → استخدم **New Order - Items**
- السيشن فارغ (لا يوجد أصناف) → استخدم **Ask Information**

**🔍 فهم السياق:**

**السيناريو 1: العميل في مرحلة الأصناف (stage: items)**
- السؤال المتوقع: "هل تريد إضافة المزيد أم إرسال موقعك؟"

**إذا قال "نعم" أو "yes" أو "أيوه":**
→ Intent: **Ask Information** (NOT Continue Order!)
→ Reply: "تمام! ما الذي تريد إضافته؟"
→ Action: null

**إذا قال "لا" أو "تمام" أو "خلاص" أو "ماشي":**
→ Intent: **Continue Order**
→ Reply: "تمام! أرسل موقعك 📍"
→ Action: "update_session"
→ Data: {"stage": "address"}

**السيناريو 2: العميل في مرحلة العنوان (stage: address)**
- السؤال المتوقع: "اختر طريقة الدفع..."
→ Intent: **Continue Order**
→ Reply: "اختر طريقة الدفع:\n\n1️⃣ كاش 💵\n2️⃣ بطاقة 💳"

**السيناريو 3: العميل في مرحلة الدفع (stage: payment)**
- السؤال المتوقع: "هل تريد تأكيد الطلب؟"
- إذا قال "نعم" → Intent: **Create Order** (NOT Continue Order!)

**⚠️ CRITICAL: Check Items First!**
Before allowing to continue, you MUST check:
1. Look at "Order Items" section in the session context
2. If it shows "(no items yet)" or "0 items" → **CANNOT** continue
3. If it shows items → **CAN** continue

**الكلمات المفتاحية للمتابعة:**
- "تمام"، "خلاص"، "ماشي"، "أكمل"، "كمل"
- "لا" (بعد سؤال إضافة المزيد)
- "أرسل الموقع"، "send location"

**الكلمات التي لا تعني المتابعة:**
- "نعم"، "yes"، "أيوه" → هذه تعني **إضافة المزيد** (استخدم Ask Information)

**الإجراء:**
1. Check current stage
2. Check if session has items
3. If NO items → Ask to add items first (Ask Information)
4. If has items:
   - If customer says "نعم/yes" → Ask what to add (Ask Information)
   - If customer says "تمام/خلاص/لا" → Ask for location (Continue Order)

**JSON Response Format (Empty Session):**
```json
{
  "intent": "Ask Information",
  "reply": "ما الذي تريد طلبه من المنيو؟ 😊",
  "action": null,
  "data": null
}
```

**JSON Response Format (Customer says "نعم" - wants to add more):**
```json
{
  "intent": "Ask Information",
  "reply": "تمام! ما الذي تريد إضافته؟ 😊",
  "action": null,
  "data": null
}
```

**JSON Response Format (Customer says "تمام" - wants to continue):**
```json
{
  "intent": "Continue Order",
  "reply": "تمام! أرسل موقعك 📍",
  "action": "update_session",
  "data": {
    "stage": "address"
  }
}
```

**⚠️ IMPORTANT:**
- "نعم" = Add more items (Ask Information)
- "تمام/خلاص" = Continue to next stage (Continue Order)
- Always check stage and items before deciding!
- NEVER use Continue Order for "نعم" in items stage!
"""

try:
    result = supabase.table('prompt_sections') \
        .update({'content': intent_continue_fixed.strip()}) \
        .eq('section_key', 'intent_continue_order') \
        .execute()
    
    if result.data:
        print("✅ Updated intent_continue_order")
    else:
        print("❌ Failed to update")
except Exception as e:
    print(f"❌ Error: {e}")

print()
print("="*70)
print("✅ Fix applied!")
print()
print("📋 Changes:")
print("   • Made it VERY clear: 'نعم' = Ask Information (add more)")
print("   • Made it VERY clear: 'تمام/خلاص' = Continue Order")
print("   • Added explicit warning: NEVER use Continue Order for 'نعم'")
print()
print("🎯 Now the bot will:")
print("   1. 'نعم' → Ask Information (ما الذي تريد إضافته؟)")
print("   2. 'تمام' → Continue Order (أرسل موقعك)")
print("   3. Keep items in session when customer says 'نعم'")

