#!/usr/bin/env python3
"""
Fix 'Yes' Ambiguity in Confirmations
Updates prompts to better handle 'نعم' responses based on context
"""
import os
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))

print("="*70)
print("🔧 Fixing 'Yes' ambiguity in confirmations")
print("="*70)

# Update intent_continue_order to handle context better
intent_continue_improved = """
**Intent: Continue Order (متابعة الطلب)**

**متى تستخدم هذه النية:**
- عندما يريد العميل **متابعة** الطلب بعد إضافة الأصناف
- عندما يقول "تمام" أو "خلاص" أو "لا" (بعد سؤال إضافة المزيد)

**🔍 فهم السياق - مهم جداً!**

**السيناريو 1: العميل في مرحلة الأصناف (stage: items)**
- السؤال المتوقع: "هل تريد إضافة المزيد أم إرسال موقعك؟"
- إذا قال "نعم" → يريد **إضافة المزيد** (استخدم Ask Information لسؤاله عما يريد)
- إذا قال "لا" أو "خلاص" أو "تمام" → يريد **إرسال الموقع** (استخدم Continue Order)

**السيناريو 2: العميل في مرحلة العنوان (stage: address)**
- السؤال المتوقع: "اختر طريقة الدفع..."
- إذا قال "نعم" → غير واضح! اسأله عن طريقة الدفع
- إذا قال "كاش" أو "بطاقة" → يختار طريقة الدفع

**السيناريو 3: العميل في مرحلة الدفع (stage: payment)**
- السؤال المتوقع: "هل تريد تأكيد الطلب؟"
- إذا قال "نعم" → يريد **تأكيد الطلب** (استخدم Create Order)
- إذا قال "لا" → يريد التعديل

**⚠️ CRITICAL: Check Items First!**
Before allowing to continue, you MUST check:
1. Look at "Order Items" section in the session context
2. If it shows "(no items yet)" or "0 items" → **CANNOT** continue
3. If it shows items → **CAN** continue

**Example 1 - "نعم" في مرحلة الأصناف:**
```
Session State: stage: items
Order Items: 2 items (هريس لحم x 2)
Last Bot Message: "هل تريد إضافة المزيد أم إرسال موقعك؟"
Customer: "نعم"
```
→ العميل يريد **إضافة المزيد**
→ Intent: "Ask Information"
→ Reply: "تمام! ما الذي تريد إضافته؟"

**Example 2 - "تمام" في مرحلة الأصناف:**
```
Session State: stage: items
Order Items: 2 items (هريس لحم x 2)
Last Bot Message: "هل تريد إضافة المزيد أم إرسال موقعك؟"
Customer: "تمام"
```
→ العميل يريد **إرسال الموقع**
→ Intent: "Continue Order"
→ Reply: "تمام! أرسل موقعك 📍"

**Example 3 - "نعم" في مرحلة الدفع:**
```
Session State: stage: payment
Order Items: 2 items (هريس لحم x 2)
Payment Method: cash
Last Bot Message: "هل تريد تأكيد الطلب؟"
Customer: "نعم"
```
→ العميل يريد **تأكيد الطلب**
→ Intent: "Create Order" (NOT Continue Order!)
→ Action: "create_order"

**الكلمات المفتاحية:**
- **للمتابعة**: "تمام"، "خلاص"، "ماشي"، "أكمل"، "كمل"، "لا" (بعد سؤال إضافة المزيد)
- **للإضافة**: "نعم"، "أيوه"، "yes" (بعد سؤال إضافة المزيد)
- **للتأكيد**: "نعم"، "أكد"، "confirm" (في مرحلة الدفع)

**⚠️ لا تستخدم هذه النية إذا:**
- العميل يقول "نعم" في مرحلة الأصناف → استخدم Ask Information
- العميل يقول "نعم" في مرحلة الدفع → استخدم Create Order (ليس Continue!)
- العميل يطلب صنف جديد → استخدم New Order - Items

**الإجراء:**
1. Check current stage
2. Check if session has items
3. If NO items → Ask to add items first
4. If has items:
   - If stage = 'items' AND customer says "تمام/خلاص/لا" → Ask for location
   - If stage = 'items' AND customer says "نعم" → Ask what to add (use Ask Information)
   - If stage = 'address' → Ask for payment method
   - If stage = 'payment' AND customer says "نعم" → Create order (use Create Order intent!)

**JSON Response Format (Empty Session):**
```json
{
  "intent": "Continue Order",
  "reply": "عذراً، لا يوجد أصناف في الطلب حالياً 😊\n\nما الذي تريد طلبه من المنيو؟",
  "action": null,
  "data": null
}
```

**JSON Response Format (Items Stage - Continue):**
```json
{
  "intent": "Continue Order",
  "reply": "تمام! أرسل موقعك 📍",
  "action": "update_session",
  "data": {
    "stage": "address"
  }
}
```

**JSON Response Format (Items Stage - Add More):**
```json
{
  "intent": "Ask Information",
  "reply": "تمام! ما الذي تريد إضافته؟ 😊",
  "action": null,
  "data": null
}
```

**JSON Response Format (Address Stage):**
```json
{
  "intent": "Continue Order",
  "reply": "اختر طريقة الدفع:\n\n1️⃣ كاش عند الاستلام 💵\n2️⃣ بطاقة ائتمانية 💳",
  "action": "update_session",
  "data": {
    "stage": "payment"
  }
}
```

**⚠️ IMPORTANT:**
- In payment stage, if customer says "نعم" to confirm → Use "Create Order" intent, NOT "Continue Order"!
- Always check the stage and last bot message to understand context
- "نعم" in items stage = add more (Ask Information)
- "تمام/خلاص" in items stage = send location (Continue Order)
- "نعم" in payment stage = confirm order (Create Order)
"""

try:
    result = supabase.table('prompt_sections') \
        .update({'content': intent_continue_improved.strip()}) \
        .eq('section_key', 'intent_continue_order') \
        .execute()
    
    if result.data:
        print("✅ Updated intent_continue_order with context-aware logic")
    else:
        print("❌ Failed to update intent_continue_order")
except Exception as e:
    print(f"❌ Error: {e}")

print()
print("="*70)
print("✅ All updates completed!")
print()
print("📋 Changes:")
print("   • Updated intent_continue_order with context-aware logic")
print("   • Added clear rules for 'نعم' vs 'تمام'")
print("   • Added examples for each scenario")
print()
print("🎯 Now the bot will:")
print("   1. Check stage before interpreting 'نعم'")
print("   2. In items stage: 'نعم' = add more, 'تمام' = continue")
print("   3. In payment stage: 'نعم' = confirm order")
print("   4. Always consider context when responding")

