#!/usr/bin/env python3
"""
Fix Similar Items Handling
Updates the prompt to handle items with similar names but different variants
"""
import os
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))

# Updated intent_new_order_items with similar items handling
new_content = """
**⚠️ قاعدة مهمة للعروض:**
- عندما يطلب العميل **عرض** (مثل "عرض الوجبة العائلية" أو "Family Meal Deal"):
  - استخدم كود العرض من المنيو (OFFER_BF065598 للوجبة العائلية، OFFER_C6889546 للغداء الخاص)
  - استخدم **سعر العرض** الموجود في المنيو (31.0 OMR للوجبة العائلية، 12.0 OMR للغداء الخاص)
  - **لا تحسب** الأصناف الفردية! العرض صنف واحد بسعر واحد
  - أضف العرض كصنف واحد في items[]

**مثال صحيح:**
```json
{
  "items": [
    {
      "code": "OFFER_BF065598",
      "name": "عرض الوجبة العائلية",
      "quantity": 1,
      "price": 31.0
    }
  ],
  "subtotal": 31.0
}
```

---

**🔍 قاعدة مهمة للأصناف المتشابهة:**

**عندما يطلب العميل صنف بدون تحديد النوع:**
- مثال: "أريد هريس" (بدون تحديد لحم أو دجاج)
- مثال: "بغيت مصلونة" (بدون تحديد لحم أو دجاج أو سمك)
- مثال: "أضف شوربة" (بدون تحديد النوع)

**يجب عليك:**
1. **البحث** في القائمة عن جميع الأصناف المتشابهة
2. **عرض الخيارات** المتاحة مع الأسعار
3. **طلب التوضيح** من العميل

**أمثلة للأصناف المتشابهة:**
- **هريس**: هريس لحم (2.8 ر.ع)، هريس دجاج (2.5 ر.ع)
- **مصلونة**: مصلونة لحم (3.8 ر.ع)، مصلونة دجاج (3.5 ر.ع)، مصلونة سمك تونة (3.5 ر.ع)
- **لحم جمل**: لحم جمل مع رز أبيض (3.8 ر.ع)، لحم جمل قصابية (3.8 ر.ع)
- **شوربة**: شوربة عدس (1.0 ر.ع)، شوربة ذرة بالدجاج (1.0 ر.ع)، شوربة هريس باللحم (1.0 ر.ع)، شوربة مشروم بالكريمة (1.5 ر.ع)، شوربة بايبلو بالروبيان (1.5 ر.ع)
- **سلطة**: سلطة عربية (1.5 ر.ع)، سلطة جرجير (1.5 ر.ع)، سلطة يونانية بجبن الفيتا (1.8 ر.ع)

**مثال - الرد الصحيح:**
```
لدينا أنواع من الهريس:

1. هريس لحم - 2.800 ر.ع
2. هريس دجاج - 2.500 ر.ع

أي نوع تفضل؟
```

**⚠️ لا تضف الصنف للطلب حتى يحدد العميل النوع!**

**JSON Response Format للأصناف المتشابهة:**
```json
{
  "intent": "Ask Information",
  "reply": "لدينا أنواع من [الصنف]:\n\n1. [النوع الأول] - [السعر] ر.ع\n2. [النوع الثاني] - [السعر] ر.ع\n\nأي نوع تفضل؟",
  "action": null,
  "data": null
}
```

---

**New Order - Items (إضافة أصناف للطلب)**

**متى تستخدم هذه النية:**
- عندما يريد العميل **إضافة** صنف **محدد بالكامل**: "أريد هريس لحم"، "أضف كبسة دجاج"
- عندما يريد العميل **طلب** صنف **محدد**: "بغيت هريس دجاج"، "طيب أريد 2 هريس لحم"
- عندما يحدد **كمية** لصنف **محدد**: "2 هريس لحم"، "3 كبسة دجاج"

**⚠️ شرط مهم:**
- يجب أن يكون الصنف **محدد بالكامل** (مثل "هريس لحم" وليس فقط "هريس")
- إذا لم يحدد العميل النوع، استخدم "Ask Information" لعرض الخيارات

**الكلمات المفتاحية للطلب:**
- "أريد"، "بغيت"، "أبغى"، "أبي"
- "أضف"، "زيد"، "حط"
- "طيب أريد"، "خلاص أريد"، "ماشي أريد"
- أرقام + اسم صنف: "2 هريس لحم"، "3 كبسة"

**أمثلة:**
- "أريد هريس لحم" → طلب (أضف 1 هريس لحم)
- "أضف كبسة دجاج" → طلب (أضف 1 كبسة دجاج)
- "طيب أريد 2 هريس لحم" → طلب (أضف 2 هريس لحم)
- "بغيت 3 كبسة" → طلب (أضف 3 كبسة دجاج)
- "أريد هريس" → **لا تضف!** استخدم Ask Information لعرض الخيارات

**⚠️ لا تستخدم هذه النية إذا:**
- العميل يسأل عن **توفر**: "عندكم هريس؟" → استخدم Ask Information
- العميل يسأل عن **سعر**: "كم سعر الهريس؟" → استخدم Ask Information
- العميل يسأل عن **وصف**: "ما هو الهريس؟" → استخدم Ask Information
- العميل لم يحدد **النوع**: "أريد هريس" → استخدم Ask Information لعرض الخيارات

**الإجراء:**
- أضف الصنف للطلب
- احسب المجموع
- اعرض ملخص الطلب
- اسأل إذا كان يريد إضافة المزيد

**🔑 قاعدة الأسماء:**
- **دائماً** استخدم اسم الصنف من قاعدة البيانات (name_ar أو name_en)
- **لا تستخدم** اسم الصنف من رسالة العميل
- **ابحث** عن الصنف في القائمة واستخدم الاسم الرسمي

**مثال:**
- العميل: "أريد حريس لحم" (خطأ إملائي)
- الرد: "تم إضافة 1 × **هريس لحم**" (الاسم الصحيح من القائمة)

---

**🚨 CRITICAL: Reply Formatting for Offers**

When customer orders an OFFER, your `reply` message MUST show:
- ✅ The offer as ONE item (not individual items)
- ✅ The offer price (31.0 OMR or 12.0 OMR)
- ❌ DO NOT list individual items in the offer
- ❌ DO NOT calculate individual prices

**Example - CORRECT reply:**
```
تم إضافة عرض الوجبة العائلية.

📋 ملخص طلبك:
- 1 × عرض الوجبة العائلية = 31.000 ر.ع

💰 المجموع الفرعي: 31.000 ر.ع

هل تريد إضافة المزيد؟
```

---

**JSON Response Format:**
```json
{
  "intent": "New Order - Items",
  "reply": "تم إضافة [quantity] × [item_name] ([price] ر.ع للوحدة) = [total] ر.ع\n\n💰 المجموع الفرعي: [subtotal] ر.ع\n\nهل تريد إضافة المزيد أم إرسال موقعك؟",
  "action": "update_session",
  "data": {
    "items": [
      {
        "code": "ITEM_CODE",
        "name": "اسم الصنف من القائمة",
        "quantity": 1,
        "price": 2.8
      }
    ],
    "subtotal": 2.8
  }
}
```

**⚠️ IMPORTANT:**
- **ALWAYS** set `"action": "update_session"` when adding items
- **ALWAYS** use item name from menu (name_ar or name_en)
- This saves the items to the database
- Without this action, items will NOT be saved!
"""

print("="*70)
print("🔧 Updating intent_new_order_items prompt")
print("="*70)

try:
    # Update the prompt section
    result = supabase.table('prompt_sections') \
        .update({'content': new_content.strip()}) \
        .eq('section_key', 'intent_new_order_items') \
        .execute()
    
    if result.data:
        print("✅ Updated intent_new_order_items")
        print()
        print("📋 Changes:")
        print("   • Added similar items handling")
        print("   • Added rule to show options when item is ambiguous")
        print("   • Added rule to always use item name from database")
        print("   • Added examples for similar items (هريس، مصلونة، شوربة، سلطة)")
        print()
        print("🎯 Now the bot will:")
        print("   1. Detect when customer orders ambiguous item (e.g., 'هريس')")
        print("   2. Show all available variants with prices")
        print("   3. Ask customer to specify which variant they want")
        print("   4. Always use official item names from database")
    else:
        print("❌ Failed to update")
        
except Exception as e:
    print(f"❌ Error: {e}")

