"""
Auto-fix order query issues in the codebase
IMPORTANT: This script will modify files. Make sure you have backups!
"""
import os
import re
from pathlib import Path
import shutil

# Mapping of incorrect → correct column names
FIXES = {
    # In .select() queries for orders table
    'select_orders': [
        (r"\.select\(['\"]([^'\"]*\b)order_number(\b[^'\"]*)['\"]", r".select('\1order_id\2'"),
        (r"\.select\(['\"]([^'\"]*\b)total(\b[^'\"]*)['\"]", r".select('\1total_omr\2'"),
        (r"\.select\(['\"]([^'\"]*\b)items(\b[^'\"]*)['\"]", r".select('\1order_items\2'"),
        (r"\.select\(['\"]([^'\"]*\b)subtotal(\b[^'\"]*)['\"]", r".select('\1subtotal_omr\2'"),
    ],
    
    # In .eq() queries
    'eq': [
        (r"\.eq\(['\"]order_number['\"]\s*,", r".eq('order_id',"),
    ],
    
    # Dictionary access and .get()
    'dict_get': [
        (r"\['order_number'\]", r"['order_id']"),
        (r'\.get\(["\']order_number["\']\s*[,\)]', r".get('order_id',"),
        (r".get\('order_number',", r".get('order_id',"),
    ],
}

def print_section(title):
    """Print section header"""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80)

def backup_file(file_path):
    """Create backup of file"""
    backup_path = str(file_path) + '.backup'
    shutil.copy2(file_path, backup_path)
    return backup_path

def fix_file(file_path, dry_run=True):
    """Fix issues in a single file"""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        original_content = content
        changes_made = []
        
        # Apply fixes for .select() queries
        for pattern, replacement in FIXES['select_orders']:
            matches = list(re.finditer(pattern, content))
            if matches:
                content = re.sub(pattern, replacement, content)
                changes_made.append(f"Fixed .select() query: {len(matches)} occurrence(s)")
        
        # Apply fixes for .eq() queries
        for pattern, replacement in FIXES['eq']:
            matches = list(re.finditer(pattern, content))
            if matches:
                content = re.sub(pattern, replacement, content)
                changes_made.append(f"Fixed .eq() query: {len(matches)} occurrence(s)")
        
        # Apply fixes for dictionary access and .get()
        for pattern, replacement in FIXES['dict_get']:
            matches = list(re.finditer(pattern, content))
            if matches:
                content = re.sub(pattern, replacement, content)
                changes_made.append(f"Fixed dict/get access: {len(matches)} occurrence(s)")
        
        # If changes were made
        if content != original_content:
            if not dry_run:
                # Backup original file
                backup_path = backup_file(file_path)
                
                # Write fixed content
                with open(file_path, 'w', encoding='utf-8') as f:
                    f.write(content)
                
                return True, changes_made, backup_path
            else:
                return True, changes_made, None
        
        return False, [], None
        
    except Exception as e:
        print(f"   ❌ Error processing {file_path}: {e}")
        return False, [], None

def fix_codebase(dry_run=True):
    """Fix all order query issues in the codebase"""
    
    mode = "DRY RUN" if dry_run else "LIVE MODE"
    print_section(f"🔧 FIXING ORDER QUERIES - {mode}")
    
    if dry_run:
        print("\n⚠️  DRY RUN MODE: No files will be modified")
        print("   Run with dry_run=False to apply fixes")
    else:
        print("\n⚠️  LIVE MODE: Files will be modified!")
        print("   Backups will be created with .backup extension")
    
    # Files to fix (based on audit results)
    files_to_fix = [
        'app/routes/webhook.py',
        'app/services/session_manager.py',
        'app/services/order_manager.py',
        'app/state_machine/order_state_manager.py',
        'scripts/verify_last_customer.py',
    ]
    
    fixed_files = []
    total_changes = 0
    
    for file_path in files_to_fix:
        path = Path(file_path)
        if not path.exists():
            print(f"\n⚠️  File not found: {file_path}")
            continue
        
        print(f"\n📄 Processing {file_path}...")
        
        has_changes, changes, backup_path = fix_file(path, dry_run=dry_run)
        
        if has_changes:
            fixed_files.append(file_path)
            total_changes += len(changes)
            
            for change in changes:
                print(f"   ✅ {change}")
            
            if backup_path:
                print(f"   💾 Backup: {backup_path}")
        else:
            print(f"   ℹ️  No changes needed")
    
    # Summary
    print_section("📊 SUMMARY")
    print(f"\n   Files processed: {len(files_to_fix)}")
    print(f"   Files modified: {len(fixed_files)}")
    print(f"   Total changes: {total_changes}")
    
    if dry_run:
        print(f"\n   ℹ️  This was a DRY RUN - no files were modified")
        print(f"   ℹ️  Run with dry_run=False to apply fixes")
    else:
        print(f"\n   ✅ Files have been fixed!")
        print(f"   💾 Backups saved with .backup extension")
        print(f"\n   ⚠️  IMPORTANT: Test the application before deleting backups!")

if __name__ == "__main__":
    import sys
    
    # Check if --apply flag is provided
    apply_fixes = '--apply' in sys.argv
    
    if apply_fixes:
        print("\n⚠️  WARNING: This will modify your files!")
        response = input("Are you sure you want to continue? (yes/no): ")
        if response.lower() != 'yes':
            print("Aborted.")
            sys.exit(0)
        
        fix_codebase(dry_run=False)
    else:
        print("\n💡 Running in DRY RUN mode...")
        print("   Add --apply flag to actually modify files")
        fix_codebase(dry_run=True)

