"""
Fix Order History Intent - Make OpenAI use track_order action
Fixes the bug where "طلباتي السابقة" doesn't trigger track_order action
"""
import asyncio
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from supabase import create_client
from app.config import settings
from app.utils.logger import app_logger as logger


async def fix_order_history_intent():
    """Update the track_order prompt to handle 'Order History' intent properly"""
    
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    # Enhanced prompt content with Order History handling
    new_content = """## 🔍 Intent: Track Order

**When to use**: Customer wants to check order status OR view order history

**Trigger phrases**:
- Arabic: "وين طلبي", "ما حالة طلبي", "ما حالة الطلب", "طلبي وصل", "كم باقي على طلبي", "متى يوصل طلبي", "ما اخر اوردر", "آخر طلب", "اخر طلب", "آخر أوردر", "اخر اوردر قمت بطلبه", "ما اخر طلب", "اخر طلب خاص بي", "طلباتي السابقة", "طلباتي", "سجل طلباتي", "الطلبات السابقة"
- English: "where is my order", "order status", "what's the status", "track order", "check order", "last order", "latest order", "my last order", "my latest order", "my orders", "order history", "previous orders", "past orders"

**🔴🔴🔴 ABSOLUTE RULE - READ THIS FIRST! 🔴🔴🔴**

**IF customer says "وين طلبي" or "where is my order":**
```json
{
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك..." (or "Searching for your order..."),
  "action": "track_order",  ← MUST BE "track_order", NEVER null!
  "data": null
}
```

**DO NOT:**
- ❌ Say "ليس لديك طلب" (You have no order)
- ❌ Use `"action": null`
- ❌ Look at draft_session items to decide
- ❌ Assume customer has no orders

**WHY:** The draft_session only shows INCOMPLETE orders. Customer may have COMPLETED orders in the database that you cannot see! You MUST use `action: "track_order"` to search the database!

---

**🚨 CRITICAL RULES:**
1. **NEVER** say "ليس لديك طلب" or "You don't have an order" WITHOUT calling track_order action first!
2. **NEVER** say "لإلغائه" (to cancel it) when customer asks about order STATUS - that's for CANCEL intent!
3. **ALWAYS** use `action: "track_order"` when customer asks about order status OR order history
4. **DO NOT** rely on draft_sessions to determine if customer has orders - they may have completed orders in the database!
5. The draft_session only shows INCOMPLETE orders. Customer may have COMPLETED orders that you MUST search for!
6. **🔴 IMPORTANT**: When customer asks "وين طلبي" or "where is my order", you MUST use `action: "track_order"` with `data: null` to search for their order (draft OR completed)!
7. **🔴 CRITICAL**: Even if you see empty items in draft_session, the customer may have a COMPLETED order in the database. ALWAYS use `action: "track_order"` to search!

**🔴 DISTINGUISH FROM CANCEL ORDER:**
- Track Order: "ما حالة", "وين", "آخر طلب", "طلباتي السابقة" → Customer wants to CHECK status
- Cancel Order: "ألغي", "إلغاء", "cancel" → Customer wants to CANCEL

**How to handle**:

1. **Extract Order Number** (if provided):
   - Look for numbers in the message: "10001", "2025-019", "019"
   - If you see "EXTRACTED ORDER NUMBER: XXXXX" in the context, USE IT
   - Order numbers can be in formats: "10001", "2025-019", "019"
   - Arabic numerals (٠-٩) are automatically converted to English (0-9)

2. **Response Strategy**:
   - If order number is extracted → Use it immediately for tracking with `action: "track_order"`
   - If customer asks "وين طلبي", "where is my order", "last order", "latest order", "طلباتي السابقة", or "order history" → **MUST** use `action: "track_order"` with `data: null`
   - **IMPORTANT**: "وين طلبي" means "where is my order" - customer wants to see their order (draft OR completed). ALWAYS use `action: "track_order"` with `data: null`
   - If no order number and not asking for order status → Ask customer to provide it (but still use `action: null`)
   - Always respond in the same language as customer's message

3. **Action Required**:
```json
{
  "intent": "track_order",
  "reply": "[Your response in customer's language]",
  "action": "track_order",
  "data": {
    "order_number": "[extracted order number OR null for latest order/history]"
  }
}
```

**Example Scenarios**:

Scenario 1: Arabic with Arabic numerals
```
Customer: "ما حالة طلبي رقم ١٠٠٠١"
System extracts: "10001"
Response: {
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك رقم 10001...",
  "action": "track_order",
  "data": {"order_number": "10001"}
}
```

Scenario 2: Customer asks "وين طلبي" (where is my order) ⚠️ CRITICAL!
```
Customer: "وين طلبي"
OR: "where is my order"
No specific order number
🚨 MUST use action: "track_order" with data: null

✅ CORRECT Response (Arabic):
{
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك...",
  "action": "track_order",
  "data": null
}

✅ CORRECT Response (English):
{
  "intent": "track_order",
  "reply": "Searching for your order...",
  "action": "track_order",
  "data": null
}

❌ WRONG Response #1 (DO NOT DO THIS):
{
  "intent": "track_order",
  "reply": "ليس لديك طلب حالياً",
  "action": null,  ← ❌ NEVER use null! Customer may have completed order!
  "data": null
}

❌ WRONG Response #2 (DO NOT DO THIS):
{
  "intent": "track_order",
  "reply": "ليس لديك طلب حالياً. ماذا تريد أن تطلب؟",
  "action": null,  ← ❌ NEVER assume no orders without searching!
  "data": null
}

❌ WRONG Response #3 (DO NOT DO THIS - This is for CANCEL, not TRACK):
{
  "intent": "track_order",
  "reply": "ليس لديك طلب حالياً لإلغائه",
  "action": null,
  "data": null
}
```

Scenario 3: Customer asks for order history ⚠️ CRITICAL!
```
Customer: "طلباتي السابقة"
OR: "my previous orders"
OR: "order history"

🚨 MUST use action: "track_order" with data: null
Response (Arabic): {
  "intent": "track_order",
  "reply": "جاري البحث عن طلباتك السابقة...",
  "action": "track_order",
  "data": null
}

Response (English): {
  "intent": "track_order",
  "reply": "Searching for your previous orders...",
  "action": "track_order",
  "data": null
}

❌ WRONG Response (DO NOT DO THIS):
{
  "intent": "Order History",
  "reply": "ليس لديك طلبات سابقة",
  "action": null,  ← ❌ NEVER use null action for order history!
  "data": null
}
```

Scenario 4: Customer sends only number (CHECK CONVERSATION HISTORY!)
```
Customer: "10001"
Context: Check recent conversation history!
- If previous message was about tracking → Intent is "track_order"
- If previous message was about canceling → Intent is "cancel_order"
- If no context → Ask what they want to do

Response (if previous was tracking): {
  "intent": "track_order",
  "reply": "جاري البحث عن طلبك رقم 10001...",
  "action": "track_order",
  "data": {"order_number": "10001"}
}
```

Scenario 5: English request
```
Customer: "where is my order 10001"
Response: {
  "intent": "track_order",
  "reply": "Searching for your order 10001...",
  "action": "track_order",
  "data": {"order_number": "10001"}
}
```

**🔴 IMPORTANT NOTES:**

1. **Order Number Formats**:
   - Simple: "10001", "10002"
   - With year: "2025-019", "2025-020"
   - Short: "019", "020"
   - All formats are valid!

2. **Arabic Numerals**:
   - System automatically converts: ١٠٠٠١ → 10001
   - You don't need to do conversion

3. **Context Awareness**:
   - Always check conversation history
   - If customer previously asked about tracking, assume they're still tracking
   - Don't ask for order number again if it was just provided

4. **Language Consistency**:
   - Arabic message → Arabic reply
   - English message → English reply
   - Check the customer's message language, not the session language

**Error Handling**:
- If order not found: "عذراً، لم أجد طلب بهذا الرقم. تأكد من الرقم وحاول مرة أخرى."
- If order belongs to different customer: "هذا الطلب غير مرتبط بحسابك."
"""

    try:
        # Update the prompt section
        response = supabase.table('prompt_sections') \
            .update({'content': new_content}) \
            .eq('section_key', 'intent_track_order') \
            .execute()
        
        if response.data:
            logger.info("✅ Successfully updated intent_track_order prompt")
            print("✅ Track Order prompt updated successfully!")
            print(f"Updated {len(response.data)} row(s)")
            print("\n📋 Changes made:")
            print("   1. ✅ Added 'طلباتي السابقة' to trigger phrases")
            print("   2. ✅ Added 'order history' to trigger phrases")
            print("   3. ✅ Added Scenario 3 for order history handling")
            print("   4. ✅ Emphasized MUST use action: 'track_order' with data: null")
            print("   5. ✅ Added wrong response examples to avoid")
            print("\n🎯 This fixes the bug where OpenAI returns action: null for 'طلباتي السابقة'")
        else:
            logger.warning("⚠️ No rows updated - section might not exist")
            print("⚠️ No rows updated. Creating new section...")
            
            # Create new section if it doesn't exist
            new_section = {
                'section_key': 'intent_track_order',
                'section_name': 'Track Order Intent',
                'content': new_content,
                'priority': 60,
                'is_required': False,
                'active': True
            }
            
            response = supabase.table('prompt_sections').insert(new_section).execute()
            
            if response.data:
                logger.info("✅ Created new intent_track_order section")
                print("✅ New section created successfully!")
            else:
                logger.error("❌ Failed to create section")
                print("❌ Failed to create section")
        
    except Exception as e:
        logger.error(f"Error updating prompt: {e}")
        print(f"❌ Error: {e}")
        raise


if __name__ == "__main__":
    print("="*80)
    print("🔧 Fixing Order History Intent")
    print("="*80)
    print("\n📝 Problem:")
    print("   When customer asks 'طلباتي السابقة', OpenAI returns:")
    print("   {")
    print('     "intent": "Order History",')
    print('     "action": null,  ← ❌ Should be "track_order"')
    print('     "data": null')
    print("   }")
    print("\n✅ Solution:")
    print("   Update prompt to explicitly require action: 'track_order' for order history")
    print("\n" + "="*80)
    print("🔄 Updating prompt in database...")
    print("="*80 + "\n")
    
    asyncio.run(fix_order_history_intent())
    
    print("\n" + "="*80)
    print("✅ Done!")
    print("="*80)
    print("\n📋 Next steps:")
    print("   1. Server will auto-reload (--reload flag)")
    print("   2. Test from WhatsApp: 'طلباتي السابقة'")
    print("   3. Expected: Bot searches database and shows order 10001")
    print("\n" + "="*80)

