"""
Fix intent_new_order_items to include action: "update_session"
"""
import os
import sys
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))

print("="*70)
print("🔧 Fixing intent_new_order_items")
print("="*70)
print()

# Get current content
result = supabase.table('prompt_sections').select('content').eq('section_key', 'intent_new_order_items').execute()

if not result.data:
    print("❌ Section not found")
    sys.exit(1)

current_content = result.data[0]['content']

# Check if already has action
if '"action": "update_session"' in current_content:
    print("✅ Section already has action: update_session")
    sys.exit(0)

# Add action instruction
new_section = '''
**JSON Response Format:**
```json
{
  "intent": "New Order - Items",
  "reply": "تم إضافة [quantity] × [item_name] ([price] ر.ع للوحدة) = [total] ر.ع\\n\\n💰 المجموع الفرعي: [subtotal] ر.ع\\n\\nهل تريد إضافة المزيد أم إرسال موقعك؟",
  "action": "update_session",
  "data": {
    "items": [
      {
        "code": "ITEM_CODE",
        "name": "اسم الصنف",
        "quantity": 1,
        "price": 2.8
      }
    ],
    "subtotal": 2.8
  }
}
```

**⚠️ IMPORTANT:**
- **ALWAYS** set `"action": "update_session"` when adding items
- This saves the items to the database
- Without this action, items will NOT be saved!
'''

# Find where to insert (after examples, before end)
if '**JSON Response Format:**' in current_content:
    # Already has format section, replace it
    start_idx = current_content.find('**JSON Response Format:**')
    # Find next section or end
    end_markers = ['\n\n**', '\n---\n', '\n\n##']
    end_idx = len(current_content)
    for marker in end_markers:
        idx = current_content.find(marker, start_idx + 100)
        if idx > start_idx and idx < end_idx:
            end_idx = idx
    
    updated_content = current_content[:start_idx] + new_section + current_content[end_idx:]
else:
    # Add at the end
    updated_content = current_content + '\n\n' + new_section

# Update database
try:
    supabase.table('prompt_sections').update({
        'content': updated_content
    }).eq('section_key', 'intent_new_order_items').execute()
    
    print("✅ Updated intent_new_order_items")
    print()
    print("📋 Added:")
    print('   - action: "update_session"')
    print('   - Instruction to ALWAYS include action')
    print()
    print("🎯 Now items will be saved to database!")
    
except Exception as e:
    print(f"❌ Error: {e}")
    sys.exit(1)

