#!/usr/bin/env python3
"""
Fix Empty Session Handling
Updates prompts to handle empty sessions (no items) correctly
"""
import os
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
env_path = Path(__file__).parent.parent / '.env'
load_dotenv(dotenv_path=env_path)

supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))

print("="*70)
print("🔧 Updating prompts for empty session handling")
print("="*70)

# 1. Update context_info to emphasize empty session check
context_info_update = """
**Session Context Rules:**

1. **Check Items First:**
   - ALWAYS check if `Order Items` section shows items
   - If it shows "(no items yet)" or "0 items", the order is EMPTY
   - Empty order = No order exists

2. **Empty Session Behavior:**
   - If session has NO items:
     - Do NOT allow address submission
     - Do NOT allow payment selection
     - Do NOT allow order confirmation
     - Guide customer to add items first

3. **Non-Empty Session Behavior:**
   - If session has items:
     - Allow address submission
     - Allow payment selection
     - Allow order confirmation

**Example - Empty Session:**
```
Order Items (0 items, 0 total qty):
  (no items yet)
- subtotal: 0.000 OMR
```
→ This is an EMPTY order. Customer must add items first!

**Example - Non-Empty Session:**
```
Order Items (2 items, 3 total qty):
  1. هريس لحم (code: HAREES_MEAT) x 2 @ 2.800 OMR = 5.600 OMR
  2. كبسة دجاج (code: KABSA_CHICKEN) x 1 @ 3.500 OMR = 3.500 OMR
- subtotal: 9.100 OMR
```
→ This order has items. Customer can proceed to address/payment.
"""

try:
    result = supabase.table('prompt_sections') \
        .select('content') \
        .eq('section_key', 'context_info') \
        .execute()
    
    if result.data:
        current_content = result.data[0]['content']
        # Append the new rules
        updated_content = current_content + "\n\n" + context_info_update.strip()
        
        supabase.table('prompt_sections') \
            .update({'content': updated_content}) \
            .eq('section_key', 'context_info') \
            .execute()
        
        print("✅ Updated context_info")
    else:
        print("❌ context_info not found")
except Exception as e:
    print(f"❌ Error updating context_info: {e}")

# 2. Update intent_location to check for empty session
intent_location_update = """
**Intent: Location (إرسال الموقع)**

**متى تستخدم هذه النية:**
- عندما يرسل العميل **موقع** (location message)
- عندما يريد العميل **إرسال عنوان**: "أرسل لك الموقع"، "هذا عنواني"

**⚠️ CRITICAL: Check Items First!**
Before accepting location, you MUST check:
1. Look at "Order Items" section in the session context
2. If it shows "(no items yet)" or "0 items" → **REJECT** the location
3. If it shows items → **ACCEPT** the location

**Example - REJECT (Empty Session):**
```
Order Items (0 items, 0 total qty):
  (no items yet)
- subtotal: 0.000 OMR
```
→ Customer Message: "أرسل لك الموقع"
→ Your Response: "عذراً، لا يوجد طلب حالياً 😊\n\nيرجى إضافة أصناف من المنيو أولاً."

**Example - ACCEPT (Has Items):**
```
Order Items (2 items, 3 total qty):
  1. هريس لحم x 2 = 5.600 OMR
- subtotal: 5.600 OMR
```
→ Customer Message: "أرسل لك الموقع"
→ Your Response: "تمام! أرسل موقعك 📍"

**الكلمات المفتاحية:**
- "أرسل لك الموقع"، "هذا عنواني"، "الموقع"
- "Location"، "Address"، "My location"

**⚠️ لا تستخدم هذه النية إذا:**
- العميل يسأل عن **التوصيل**: "هل توصلون؟" → استخدم Ask Information
- العميل يسأل عن **مناطق التوصيل**: "أين توصلون؟" → استخدم Ask Information

**الإجراء:**
1. Check if session has items (MUST have items!)
2. If NO items → Reject and ask to add items first
3. If has items → Accept and ask for location
4. Update stage to 'address'

**JSON Response Format (Empty Session):**
```json
{
  "intent": "Location",
  "reply": "عذراً، لا يوجد طلب حالياً 😊\n\nيرجى إضافة أصناف من المنيو أولاً.\n\nما الذي تريد طلبه؟",
  "action": null,
  "data": null
}
```

**JSON Response Format (Has Items):**
```json
{
  "intent": "Location",
  "reply": "تمام! أرسل موقعك 📍",
  "action": "update_session",
  "data": {
    "stage": "address"
  }
}
```
"""

try:
    result = supabase.table('prompt_sections') \
        .update({'content': intent_location_update.strip()}) \
        .eq('section_key', 'intent_location') \
        .execute()
    
    if result.data:
        print("✅ Updated intent_location")
    else:
        print("❌ Failed to update intent_location")
except Exception as e:
    print(f"❌ Error updating intent_location: {e}")

# 3. Update intent_continue_order to check for empty session
intent_continue_update = """
**Intent: Continue Order (متابعة الطلب)**

**متى تستخدم هذه النية:**
- عندما يريد العميل **متابعة** الطلب: "تمام"، "خلاص"، "ماشي"، "نعم"، "أكمل"
- عندما ينتهي من إضافة الأصناف ويريد الانتقال للمرحلة التالية

**⚠️ CRITICAL: Check Items First!**
Before allowing to continue, you MUST check:
1. Look at "Order Items" section in the session context
2. If it shows "(no items yet)" or "0 items" → **CANNOT** continue
3. If it shows items → **CAN** continue

**Example - CANNOT Continue (Empty Session):**
```
Order Items (0 items, 0 total qty):
  (no items yet)
```
→ Customer: "تمام"
→ Your Response: "عذراً، لا يوجد أصناف في الطلب حالياً 😊\n\nما الذي تريد طلبه؟"

**Example - CAN Continue (Has Items):**
```
Order Items (2 items, 3 total qty):
  1. هريس لحم x 2 = 5.600 OMR
```
→ Customer: "تمام"
→ Your Response: "تمام! أرسل موقعك 📍"

**الكلمات المفتاحية:**
- "تمام"، "خلاص"، "ماشي"، "أكمل"، "كمل"
- "نعم"، "أيوه"، "yes"، "ok"، "okay"
- "أرسل الموقع"، "send location"

**الإجراء:**
1. Check current stage
2. Check if session has items
3. If NO items → Ask to add items first
4. If has items:
   - If stage = 'items' → Ask for location
   - If stage = 'address' → Ask for payment method
   - If stage = 'payment' → Confirm order

**JSON Response Format (Empty Session):**
```json
{
  "intent": "Continue Order",
  "reply": "عذراً، لا يوجد أصناف في الطلب حالياً 😊\n\nما الذي تريد طلبه من المنيو؟",
  "action": null,
  "data": null
}
```

**JSON Response Format (Has Items - Items Stage):**
```json
{
  "intent": "Continue Order",
  "reply": "تمام! أرسل موقعك 📍",
  "action": "update_session",
  "data": {
    "stage": "address"
  }
}
```
"""

try:
    result = supabase.table('prompt_sections') \
        .update({'content': intent_continue_update.strip()}) \
        .eq('section_key', 'intent_continue_order') \
        .execute()
    
    if result.data:
        print("✅ Updated intent_continue_order")
    else:
        print("❌ Failed to update intent_continue_order")
except Exception as e:
    print(f"❌ Error updating intent_continue_order: {e}")

print()
print("="*70)
print("✅ All prompts updated successfully!")
print()
print("📋 Changes:")
print("   • Added empty session check to context_info")
print("   • Updated intent_location to reject empty sessions")
print("   • Updated intent_continue_order to reject empty sessions")
print()
print("🎯 Now the bot will:")
print("   1. Check if session has items before accepting location")
print("   2. Check if session has items before allowing to continue")
print("   3. Guide customer to add items first if session is empty")

