#!/usr/bin/env python3
"""
Fix draft_sessions table - Add missing waiting_for_more_items column
"""

import os
import sys
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

from supabase import create_client

# Get settings from environment
SUPABASE_URL = os.getenv('SUPABASE_URL')
SUPABASE_KEY = os.getenv('SUPABASE_KEY')

def main():
    print("="*70)
    print("🔧 Fixing draft_sessions Table")
    print("="*70)
    
    supabase = create_client(SUPABASE_URL, SUPABASE_KEY)
    
    # 1. Check if table exists
    print("\n1️⃣ Checking if draft_sessions table exists...")
    try:
        result = supabase.table('draft_sessions').select('*').limit(1).execute()
        print(f"   ✅ Table exists with {len(result.data)} rows")
        
        if result.data:
            columns = list(result.data[0].keys())
            print(f"\n   📋 Current columns ({len(columns)}):")
            for col in sorted(columns):
                print(f"      - {col}")
            
            # Check if waiting_for_more_items exists
            if 'waiting_for_more_items' in columns:
                print(f"\n   ✅ Column 'waiting_for_more_items' already exists!")
                return
            else:
                print(f"\n   ⚠️  Column 'waiting_for_more_items' is MISSING")
        
    except Exception as e:
        print(f"   ❌ Error: {e}")
        print(f"\n   Creating draft_sessions table...")
        
        # Create table if it doesn't exist
        create_query = """
        CREATE TABLE IF NOT EXISTS draft_sessions (
            id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
            chat_id TEXT NOT NULL,
            customer_id UUID REFERENCES customers(id),
            items JSONB DEFAULT '[]'::jsonb,
            subtotal NUMERIC(10,2) DEFAULT 0,
            delivery_fee NUMERIC(10,2) DEFAULT 0,
            total NUMERIC(10,2) DEFAULT 0,
            payment_method TEXT,
            delivery_address TEXT,
            notes TEXT,
            status TEXT DEFAULT 'active',
            order_phase TEXT DEFAULT 'items',
            phase_start_time TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
            waiting_for_choice BOOLEAN DEFAULT FALSE,
            waiting_for_quantity BOOLEAN DEFAULT FALSE,
            waiting_for_more_items BOOLEAN DEFAULT FALSE,
            pending_items JSONB DEFAULT '[]'::jsonb,
            location_lat DOUBLE PRECISION,
            location_lng DOUBLE PRECISION,
            location_link TEXT,
            created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
            updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
        );
        
        CREATE INDEX IF NOT EXISTS idx_draft_sessions_chat_id ON draft_sessions(chat_id);
        CREATE INDEX IF NOT EXISTS idx_draft_sessions_status ON draft_sessions(status);
        """
        
        try:
            # Note: Supabase REST API doesn't support CREATE TABLE
            # You need to run this in Supabase SQL Editor
            print(f"\n   ⚠️  Cannot create table via REST API")
            print(f"\n   📝 Please run this SQL in Supabase SQL Editor:")
            print(f"\n{create_query}")
            return
        except Exception as e2:
            print(f"   ❌ Error creating table: {e2}")
            return
    
    # 2. Add missing column
    print(f"\n2️⃣ Adding waiting_for_more_items column...")
    
    alter_query = """
    ALTER TABLE draft_sessions 
    ADD COLUMN IF NOT EXISTS waiting_for_more_items BOOLEAN DEFAULT FALSE;
    """
    
    print(f"\n   ⚠️  Cannot alter table via REST API")
    print(f"\n   📝 Please run this SQL in Supabase SQL Editor:")
    print(f"\n{alter_query}")
    
    print("\n" + "="*70)
    print("✅ Instructions provided!")
    print("="*70)

if __name__ == "__main__":
    main()

