#!/usr/bin/env python3
"""
Export menu items from Supabase to local JSON file
"""
import json
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from supabase import create_client
from app.config import settings

def export_menu_to_json():
    """Export all menu items from Supabase to JSON file"""
    print("🔄 Connecting to Supabase...")
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    print("📥 Fetching menu items...")
    try:
        response = supabase.table('menu_items') \
            .select('*') \
            .order('code') \
            .execute()

        menu_items = response.data if response.data else []
        print(f"✅ Found {len(menu_items)} menu items")

        # Transform to local format
        local_menu = []
        for item in menu_items:
            local_item = {
                "code": item.get('code'),
                "name_ar": item.get('name_ar'),
                "name_en": item.get('name_en'),
                "category": item.get('category', 'غير مصنف'),
                "price": float(item.get('price_omr', 0)),
                "description": item.get('description', ''),
                "tags": item.get('tags', []) if item.get('tags') else [],
                "active": True  # Default to active
            }
            local_menu.append(local_item)
        
        # Create data directory if not exists
        data_dir = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), 'data')
        os.makedirs(data_dir, exist_ok=True)
        
        # Save to JSON file
        output_file = os.path.join(data_dir, 'menu_items_local.json')
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(local_menu, f, ensure_ascii=False, indent=2)
        
        print(f"✅ Menu exported successfully to: {output_file}")
        print(f"📊 Total items: {len(local_menu)}")
        print(f"📊 Active items: {sum(1 for item in local_menu if item['active'])}")
        
        # Print sample
        if local_menu:
            print("\n📋 Sample item:")
            print(json.dumps(local_menu[0], ensure_ascii=False, indent=2))
        
        return local_menu
        
    except Exception as e:
        print(f"❌ Error: {e}")
        import traceback
        traceback.print_exc()
        return []

if __name__ == "__main__":
    export_menu_to_json()

