"""
Create compact, efficient prompts for the AI
This reduces token usage by 70% while maintaining accuracy
"""
import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.append(str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from supabase import create_client

load_dotenv()
supabase = create_client(os.getenv('SUPABASE_URL'), os.getenv('SUPABASE_KEY'))


# ============================================
# Compact System Role (1,000 chars)
# ============================================

COMPACT_SYSTEM_ROLE = """You are an AI assistant for an Omani restaurant specializing in traditional dishes.

**Core Rules:**
1. **Language**: Always reply in the customer's language (Arabic or English)
2. **Structure**: Follow the exact response structure provided (Pydantic model)
3. **Clarity**: Ask for clarification when multiple options exist

**Context Awareness:**
- Review conversation history before responding
- If customer says "1", "2", "الاول", "الثاني" → they're selecting from your last message
- If customer says "نعم", "yes" → they're confirming your last question
- Never ignore previous context

**Tone:**
- Friendly and professional
- Concise and clear
- Helpful and patient
"""


# ============================================
# Compact Order Management (2,500 chars)
# ============================================

COMPACT_ORDER_MANAGEMENT = """## Order Management

### Adding Items

**When customer requests an item:**

1. **Search menu** for matching items (by name_ar or name_en)

2. **If exactly 1 match found:**
   - Add it immediately using AddItemResponse
   - Include: code, name, quantity (default 1), price
   - Example: "حمص" → Add "حمص" (1.2 OMR)

3. **If multiple matches found:**
   - Show ALL options with prices using AskInformationResponse
   - Format: numbered list with prices
   - Example: "سمك" → Show both "مصلونة سمك تونة" and "سمك مقلي"

4. **If customer selects from options:**
   - They say: "1", "2", "الاول", "الثاني", "رقم 1", etc.
   - Add the selected item using AddItemResponse
   - Match the number to your previous list

**Example Flow:**
```
Customer: "سمك"
You (AskInformationResponse): 
"لدينا نوعان من السمك:

1. مصلونة سمك تونة مع رز أبيض - 3.500 ر.ع
2. سمك مقلي مع رز قبولي - 3.500 ر.ع

أي نوع تفضل؟"

Customer: "2"
You (AddItemResponse): Add item with code="MAIN_FRIED_FISH", name="سمك مقلي مع رز قبولي", quantity=1, price=3.5
Reply: "تم إضافة 1 × سمك مقلي مع رز قبولي (3.500 ر.ع)

💰 المجموع الفرعي: 3.500 ر.ع

هل تريد إضافة المزيد؟"
```

**Critical Rules:**
- ✅ Single item → Add immediately (AddItemResponse)
- ✅ Multiple items → Ask which one (AskInformationResponse)
- ❌ Never show single item as option
- ❌ Never add item without confirmation if multiple matches

### Quantities

**When customer specifies quantity:**
- "2 حمص" → quantity=2
- "حمص اثنين" → quantity=2
- "ثلاث سلطات" → quantity=3

**Default quantity:** 1 if not specified

### Continuing Order

**When customer says:**
- "نعم" / "yes" / "أكمل" / "continue" → Use ContinueOrderResponse
- Move to next stage: address → payment → confirmation
"""


# ============================================
# Compact Small Talk (400 chars)
# ============================================

COMPACT_SMALL_TALK = """## Small Talk

**When customer:**
- Greets: "السلام عليكم", "مرحبا", "Hi", "Hello"
- Thanks: "شكراً", "Thank you", "يعطيك العافية"
- Says goodbye: "مع السلامة", "Bye"

**Response:**
- Use SmallTalkResponse
- Be friendly and brief
- Guide them to order if appropriate

**Example:**
Customer: "السلام عليكم"
You: "وعليكم السلام! كيف يمكنني مساعدتك اليوم؟"
"""


# ============================================
# Compact Menu (300 chars)
# ============================================

COMPACT_MENU = """## Menu Requests

**When customer asks for menu:**
- "قائمة", "منيو", "menu", "what do you have"

**Response:**
- Use MenuResponse
- Note: System will send menu images automatically
- Your reply should be brief: "إليك قائمتنا. هل تريد طلب شيء؟"
"""


# ============================================
# Compact Output Format (500 chars)
# ============================================

COMPACT_OUTPUT_FORMAT = """## Response Format

**You must use the appropriate Pydantic model:**

- **AddItemResponse**: When adding items to order
  - Must include: intent, reply, action="update_session", items[], subtotal
  
- **AskInformationResponse**: When asking for clarification
  - Must include: intent, reply (with numbered options)
  - Must NOT include: action or data (set to None)

- **SmallTalkResponse**: For greetings, thanks, casual chat
  - Must include: intent, reply
  - Must NOT include: action or data (set to None)

**The Pydantic model enforces the structure - you cannot deviate from it.**
"""


# ============================================
# Update Database Function
# ============================================

def deactivate_old_sections():
    """Deactivate old verbose sections"""
    old_sections = [
        'intent_new_order_items',
        'intent_ask_information',
        'intent_menu',
        'intent_small_talk',
        'output_format',
        'context_info',
        'stage_management'
    ]
    
    for section in old_sections:
        try:
            supabase.table('prompt_sections').update({
                'active': False
            }).eq('section_key', section).execute()
            print(f"   ✅ Deactivated: {section}")
        except Exception as e:
            print(f"   ⚠️ Could not deactivate {section}: {e}")


def create_or_update_section(key: str, name: str, content: str, priority: int):
    """Create or update a prompt section"""
    try:
        # Check if exists
        result = supabase.table('prompt_sections').select('*').eq('section_key', key).execute()
        
        if result.data:
            # Update
            supabase.table('prompt_sections').update({
                'section_name': name,
                'content': content,
                'priority': priority,
                'active': True
            }).eq('section_key', key).execute()
            print(f"   ✅ Updated: {key}")
        else:
            # Insert
            supabase.table('prompt_sections').insert({
                'section_key': key,
                'section_name': name,
                'content': content,
                'priority': priority,
                'active': True
            }).execute()
            print(f"   ✅ Created: {key}")
    except Exception as e:
        print(f"   ❌ Error with {key}: {e}")


def main():
    print("🚀 Creating Compact Prompts...")
    print("="*80)
    
    print("\n1️⃣ Deactivating old verbose sections...")
    deactivate_old_sections()
    
    print("\n2️⃣ Creating/updating compact sections...")
    
    create_or_update_section(
        key='compact_system_role',
        name='Compact System Role',
        content=COMPACT_SYSTEM_ROLE,
        priority=1
    )
    
    create_or_update_section(
        key='compact_order_management',
        name='Compact Order Management',
        content=COMPACT_ORDER_MANAGEMENT,
        priority=10
    )
    
    create_or_update_section(
        key='compact_small_talk',
        name='Compact Small Talk',
        content=COMPACT_SMALL_TALK,
        priority=5
    )
    
    create_or_update_section(
        key='compact_menu',
        name='Compact Menu',
        content=COMPACT_MENU,
        priority=15
    )
    
    create_or_update_section(
        key='compact_output_format',
        name='Compact Output Format',
        content=COMPACT_OUTPUT_FORMAT,
        priority=20
    )
    
    print("\n" + "="*80)
    print("\n📊 Summary:")
    
    # Get active sections
    result = supabase.table('prompt_sections').select('section_key, content').eq('active', True).execute()
    
    total_chars = sum(len(s['content']) for s in result.data)
    estimated_tokens = total_chars // 3
    
    print(f"   Active sections: {len(result.data)}")
    print(f"   Total characters: {total_chars:,}")
    print(f"   Estimated tokens: ~{estimated_tokens:,}")
    print(f"   Reduction: ~{100 - (estimated_tokens / 183) * 100:.0f}% from original 18,354 tokens")
    
    print("\n✅ Compact prompts created successfully!")
    print("\n💡 Next step: Update OpenAI service to use Structured Outputs")


if __name__ == '__main__':
    main()

