"""
Comprehensive audit of all database queries in the codebase
"""
import os
import re
from pathlib import Path

# Database schema (from actual database)
SCHEMA = {
    'orders': {
        'columns': ['id', 'order_id', 'chat_id', 'customer_id', 'status', 
                   'total_omr', 'subtotal_omr', 'delivery_fee_omr', 'discount_omr', 'tax_omr',
                   'payment_method', 'delivery_address', 'address_type', 
                   'delivery_latitude', 'delivery_longitude', 'delivery_location_name',
                   'location_lat', 'location_lng', 'location_link', 'notes',
                   'created_at', 'updated_at'],
        'wrong_names': ['order_number', 'total', 'subtotal', 'items', 'delivery_fee', 'discount', 'tax']
    },
    'order_items': {
        'columns': ['id', 'order_id', 'item_code', 'item_name', 'qty', 'unit_price_omr', 'subtotal_omr'],
        'wrong_names': ['quantity', 'price', 'unit_price', 'subtotal']
    },
    'draft_sessions': {
        'columns': ['id', 'chat_id', 'customer_id', 'stage', 'status', 'order_phase',
                   'items', 'subtotal', 'address', 'address_type', 'latitude', 'longitude',
                   'location_name', 'location_lat', 'location_lng', 'location_link',
                   'payment_method', 'notes', 'service_type', 'lang',
                   'waiting_for_quantity', 'waiting_for_item_choice', 'selected_item_code',
                   'pending_item_choices', 'active_intent', 'last_message_id',
                   'promoted_order_id', 'phase_start_time', 'stt_buffer', 'attachments',
                   'created_at', 'updated_at'],
        'wrong_names': []
    },
    'chats': {
        'columns': ['id', 'chat_id', 'customer_id', 'instance_name', 'last_seen_at'],
        'wrong_names': []
    },
    'menu_items': {
        'columns': ['code', 'name_ar', 'name_en', 'description_ar', 'description_en',
                   'price_omr', 'category', 'active', 'sort_order'],
        'wrong_names': ['price']
    },
    'conversation_history': {
        'columns': ['id', 'chat_id', 'role', 'content', 'created_at'],
        'wrong_names': []
    },
    'customers': {
        'columns': ['id', 'phone_e164', 'name', 'lang', 'last_adress', 'created_at'],
        'wrong_names': ['phone', 'address']
    }
}

def find_python_files(directory):
    """Find all Python files in directory"""
    python_files = []
    for root, dirs, files in os.walk(directory):
        # Skip __pycache__ and venv directories
        dirs[:] = [d for d in dirs if d not in ['__pycache__', 'venv', '.venv', 'node_modules']]
        for file in files:
            if file.endswith('.py'):
                python_files.append(os.path.join(root, file))
    return python_files

def extract_table_queries(file_path):
    """Extract all .table() queries from a file"""
    with open(file_path, 'r', encoding='utf-8') as f:
        content = f.read()
    
    # Find all .table('table_name') patterns
    pattern = r"\.table\(['\"](\w+)['\"]\)"
    matches = re.finditer(pattern, content)
    
    queries = []
    lines = content.split('\n')
    
    for match in matches:
        table_name = match.group(1)
        # Find line number
        line_num = content[:match.start()].count('\n') + 1
        
        # Get context (5 lines around the match)
        start_line = max(0, line_num - 3)
        end_line = min(len(lines), line_num + 3)
        context = '\n'.join(lines[start_line:end_line])
        
        queries.append({
            'table': table_name,
            'line': line_num,
            'context': context
        })
    
    return queries

def check_query_for_issues(query, file_path):
    """Check if query uses wrong column names"""
    issues = []
    table_name = query['table']
    context = query['context']
    
    if table_name not in SCHEMA:
        return issues
    
    schema = SCHEMA[table_name]
    
    # Check for wrong column names
    for wrong_name in schema['wrong_names']:
        # Look for patterns like .select('wrong_name') or .eq('wrong_name')
        patterns = [
            rf"\.select\(['\"][^'\"]*\b{wrong_name}\b[^'\"]*['\"]\)",
            rf"\.eq\(['\"]({wrong_name})['\"]\s*,",
            rf"\.update\(\{{[^}}]*['\"]({wrong_name})['\"]\s*:",
            rf"\['{wrong_name}'\]",
            rf'\.get\(["\']({wrong_name})["\']\)',
        ]
        
        for pattern in patterns:
            if re.search(pattern, context):
                issues.append({
                    'type': 'wrong_column',
                    'table': table_name,
                    'wrong_name': wrong_name,
                    'line': query['line'],
                    'context': context
                })
                break
    
    return issues

def main():
    """Main audit function"""
    print("="*80)
    print("  🔍 COMPREHENSIVE DATABASE QUERY AUDIT")
    print("="*80)
    
    # Find all Python files
    app_dir = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'app')
    python_files = find_python_files(app_dir)
    
    print(f"\n📁 Found {len(python_files)} Python files to audit\n")
    
    all_issues = []
    
    for file_path in python_files:
        rel_path = os.path.relpath(file_path, os.path.dirname(os.path.dirname(__file__)))
        queries = extract_table_queries(file_path)
        
        if not queries:
            continue
        
        file_issues = []
        for query in queries:
            issues = check_query_for_issues(query, file_path)
            file_issues.extend(issues)
        
        if file_issues:
            all_issues.extend([(rel_path, issue) for issue in file_issues])
    
    # Print results
    if all_issues:
        print("="*80)
        print(f"  ❌ FOUND {len(all_issues)} POTENTIAL ISSUES")
        print("="*80)
        
        for file_path, issue in all_issues:
            print(f"\n📄 File: {file_path}")
            print(f"   Line: {issue['line']}")
            print(f"   Table: {issue['table']}")
            print(f"   Issue: Using '{issue['wrong_name']}' (wrong column name)")
            print(f"\n   Context:")
            for line in issue['context'].split('\n'):
                print(f"      {line}")
    else:
        print("="*80)
        print("  ✅ NO ISSUES FOUND!")
        print("="*80)
        print("\n✨ All database queries are using correct column names!")
    
    print("\n" + "="*80)
    print(f"  📊 AUDIT SUMMARY")
    print("="*80)
    print(f"  • Files scanned: {len(python_files)}")
    print(f"  • Issues found: {len(all_issues)}")
    print("="*80)

if __name__ == '__main__':
    main()

