"""
Cleanup Test Data Script
Removes all test phone numbers except the specified one
"""
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from supabase import create_client
from app.config import settings

# Phone number to keep
KEEP_PHONE = "201556307784"

def cleanup_database():
    """Remove all test data except the specified phone number"""
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    print("\n" + "="*80)
    print("🗑️  DATABASE CLEANUP - REMOVING TEST DATA")
    print("="*80 + "\n")
    
    print(f"📌 Keeping phone number: {KEEP_PHONE}")
    print(f"🗑️  Removing all other phone numbers...\n")
    
    # Step 1: Get all chats
    print("Step 1: Fetching all chats...")
    try:
        all_chats = supabase.table('chats') \
            .select('id, chat_id') \
            .execute()
        
        if not all_chats.data:
            print("✅ No chats found in database")
            return
        
        print(f"   Found {len(all_chats.data)} chats")
        
        # Separate chats to keep and delete
        keep_chat_ids = []
        delete_chat_ids = []
        
        for chat in all_chats.data:
            chat_id = chat.get('chat_id', '')
            # Extract phone number (format: 201556307784@s.whatsapp.net)
            phone = chat_id.split('@')[0] if '@' in chat_id else chat_id
            
            if phone == KEEP_PHONE:
                keep_chat_ids.append(chat['id'])
                print(f"   ✅ Keeping chat: {chat_id}")
            else:
                delete_chat_ids.append(chat['id'])
                print(f"   🗑️  Will delete chat: {chat_id}")
        
        print(f"\n📊 Summary:")
        print(f"   Chats to keep: {len(keep_chat_ids)}")
        print(f"   Chats to delete: {len(delete_chat_ids)}")
        
        if not delete_chat_ids:
            print("\n✅ No chats to delete!")
            return
        
        # Step 2: Delete related orders
        print(f"\nStep 2: Deleting orders for {len(delete_chat_ids)} chats...")
        try:
            # Get orders to delete
            orders_response = supabase.table('orders') \
                .select('id, order_id, chat_id') \
                .in_('chat_id', delete_chat_ids) \
                .execute()
            
            if orders_response.data:
                order_ids = [order['id'] for order in orders_response.data]
                print(f"   Found {len(order_ids)} orders to delete")
                
                # Delete order items first
                print(f"   Deleting order items...")
                for order_id in order_ids:
                    supabase.table('order_items') \
                        .delete() \
                        .eq('order_id', order_id) \
                        .execute()
                
                # Delete orders
                print(f"   Deleting orders...")
                for order_id in order_ids:
                    supabase.table('orders') \
                        .delete() \
                        .eq('id', order_id) \
                        .execute()
                
                print(f"   ✅ Deleted {len(order_ids)} orders")
            else:
                print(f"   ℹ️  No orders found")
        except Exception as e:
            print(f"   ⚠️  Error deleting orders: {e}")
        
        # Step 3: Delete draft sessions
        print(f"\nStep 3: Deleting draft sessions...")
        try:
            sessions_response = supabase.table('draft_sessions') \
                .select('id, chat_id') \
                .in_('chat_id', delete_chat_ids) \
                .execute()
            
            if sessions_response.data:
                session_ids = [session['id'] for session in sessions_response.data]
                print(f"   Found {len(session_ids)} sessions to delete")
                
                for session_id in session_ids:
                    supabase.table('draft_sessions') \
                        .delete() \
                        .eq('id', session_id) \
                        .execute()
                
                print(f"   ✅ Deleted {len(session_ids)} sessions")
            else:
                print(f"   ℹ️  No sessions found")
        except Exception as e:
            print(f"   ⚠️  Error deleting sessions: {e}")
        
        # Step 4: Delete chats
        print(f"\nStep 4: Deleting {len(delete_chat_ids)} chats...")
        try:
            for chat_id in delete_chat_ids:
                supabase.table('chats') \
                    .delete() \
                    .eq('id', chat_id) \
                    .execute()
            
            print(f"   ✅ Deleted {len(delete_chat_ids)} chats")
        except Exception as e:
            print(f"   ⚠️  Error deleting chats: {e}")
        
        # Step 5: Verify cleanup
        print(f"\nStep 5: Verifying cleanup...")
        remaining_chats = supabase.table('chats') \
            .select('id, chat_id') \
            .execute()
        
        print(f"   Remaining chats: {len(remaining_chats.data)}")
        for chat in remaining_chats.data:
            print(f"   ✅ {chat['chat_id']}")
        
        print("\n" + "="*80)
        print("✅ CLEANUP COMPLETED SUCCESSFULLY!")
        print("="*80 + "\n")
        
    except Exception as e:
        print(f"\n❌ Error during cleanup: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    print("\n⚠️  WARNING: This will delete all test data except phone number:", KEEP_PHONE)
    response = input("Are you sure you want to continue? (yes/no): ")
    
    if response.lower() == 'yes':
        cleanup_database()
    else:
        print("\n❌ Cleanup cancelled")

