"""
Cleanup Orphan Orders
Removes orders that don't belong to the kept phone number
"""
import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from supabase import create_client
from app.config import settings

KEEP_PHONE = "201556307784"

def cleanup_orphan_orders():
    """Remove orders that don't belong to the kept phone number"""
    supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
    
    print("\n" + "="*80)
    print("🗑️  CLEANING UP ORPHAN ORDERS")
    print("="*80 + "\n")
    
    # Get the chat_id for the phone we want to keep
    print(f"Step 1: Finding chat for phone {KEEP_PHONE}...")
    keep_chat = supabase.table('chats') \
        .select('id, chat_id') \
        .like('chat_id', f'{KEEP_PHONE}%') \
        .execute()
    
    if not keep_chat.data:
        print(f"   ❌ Chat not found for {KEEP_PHONE}")
        return
    
    keep_chat_uuid = keep_chat.data[0]['id']
    print(f"   ✅ Found chat UUID: {keep_chat_uuid}")
    
    # Get all orders
    print(f"\nStep 2: Fetching all orders...")
    all_orders = supabase.table('orders') \
        .select('id, order_id, chat_id, status') \
        .execute()
    
    print(f"   Found {len(all_orders.data)} orders")
    
    # Separate orders to keep and delete
    keep_orders = []
    delete_orders = []
    
    for order in all_orders.data:
        if order['chat_id'] == keep_chat_uuid:
            keep_orders.append(order)
            print(f"   ✅ Keeping order: {order['order_id']}")
        else:
            delete_orders.append(order)
            print(f"   🗑️  Will delete order: {order['order_id']} (chat_id: {order['chat_id']})")
    
    print(f"\n📊 Summary:")
    print(f"   Orders to keep: {len(keep_orders)}")
    print(f"   Orders to delete: {len(delete_orders)}")
    
    if not delete_orders:
        print("\n✅ No orphan orders to delete!")
        return
    
    # Delete order items first
    print(f"\nStep 3: Deleting order items...")
    for order in delete_orders:
        try:
            supabase.table('order_items') \
                .delete() \
                .eq('order_id', order['id']) \
                .execute()
            print(f"   ✅ Deleted items for order {order['order_id']}")
        except Exception as e:
            print(f"   ⚠️  Error deleting items for {order['order_id']}: {e}")
    
    # Delete orders
    print(f"\nStep 4: Deleting orders...")
    for order in delete_orders:
        try:
            supabase.table('orders') \
                .delete() \
                .eq('id', order['id']) \
                .execute()
            print(f"   ✅ Deleted order {order['order_id']}")
        except Exception as e:
            print(f"   ⚠️  Error deleting order {order['order_id']}: {e}")
    
    # Verify
    print(f"\nStep 5: Verifying cleanup...")
    remaining_orders = supabase.table('orders') \
        .select('id, order_id, chat_id') \
        .execute()
    
    print(f"   Remaining orders: {len(remaining_orders.data)}")
    for order in remaining_orders.data:
        if order['chat_id'] == keep_chat_uuid:
            print(f"   ✅ Order {order['order_id']} (belongs to {KEEP_PHONE})")
        else:
            print(f"   ⚠️  Order {order['order_id']} (orphan - chat_id: {order['chat_id']})")
    
    print("\n" + "="*80)
    print("✅ CLEANUP COMPLETED!")
    print("="*80 + "\n")

if __name__ == "__main__":
    print(f"\n⚠️  WARNING: This will delete all orders except those for {KEEP_PHONE}")
    response = input("Are you sure you want to continue? (yes/no): ")
    
    if response.lower() == 'yes':
        cleanup_orphan_orders()
    else:
        print("\n❌ Cleanup cancelled")

