"""
Cleanup Database - Remove old prompt_sections table
Since we moved all prompts to server-side files, we don't need this table anymore
"""
import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from dotenv import load_dotenv
from supabase import create_client

# Load environment variables
load_dotenv()

def cleanup_database():
    """Remove old prompt_sections table"""
    
    print("\n" + "="*60)
    print("🧹 Database Cleanup - Remove old prompt_sections table")
    print("="*60 + "\n")
    
    # Initialize Supabase
    supabase = create_client(
        os.getenv('SUPABASE_URL'),
        os.getenv('SUPABASE_KEY')
    )
    
    print("⚠️  WARNING: This will permanently delete the prompt_sections table!")
    print("   All prompts are now in app/prompts/*.py files")
    print("   This table is no longer needed.\n")
    
    # Step 1: Backup first (optional)
    print("📦 Step 1: Creating backup...")
    try:
        result = supabase.table('prompt_sections').select('*').execute()
        sections = result.data if result.data else []
        
        if sections:
            print(f"   ✅ Found {len(sections)} sections in database")
            
            # Save to file as backup
            backup_file = Path(__file__).parent.parent / 'backups' / 'prompt_sections_backup.json'
            backup_file.parent.mkdir(exist_ok=True)
            
            import json
            with open(backup_file, 'w', encoding='utf-8') as f:
                json.dump(sections, f, ensure_ascii=False, indent=2)
            
            print(f"   ✅ Backup saved to: {backup_file}")
        else:
            print("   ℹ️  No sections found (table might be empty)")
            
    except Exception as e:
        print(f"   ⚠️  Could not create backup: {str(e)}")
        print("   (This is OK if table doesn't exist)")
    
    # Step 2: Ask for confirmation
    print("\n❓ Do you want to proceed with deleting the table?")
    print("   Type 'yes' to confirm, or anything else to cancel:")
    
    confirmation = input("   > ").strip().lower()
    
    if confirmation != 'yes':
        print("\n❌ Cancelled. No changes made.")
        print("="*60 + "\n")
        return
    
    # Step 3: Drop the table
    print("\n🗑️  Step 2: Dropping prompt_sections table...")
    try:
        # Use raw SQL to drop table
        # Note: This requires admin privileges
        supabase.postgrest.session.execute_sql(
            "DROP TABLE IF EXISTS prompt_sections CASCADE;"
        )
        print("   ✅ Table dropped successfully!")
        
    except Exception as e:
        print(f"   ⚠️  Could not drop table using Supabase client: {str(e)}")
        print("\n   📝 Manual SQL command:")
        print("   " + "-"*56)
        print("   DROP TABLE IF EXISTS prompt_sections CASCADE;")
        print("   " + "-"*56)
        print("\n   Please run this command in Supabase SQL Editor:")
        print("   https://supabase.com/dashboard/project/YOUR_PROJECT/sql")
    
    # Step 4: Verify
    print("\n✅ Step 3: Verification...")
    try:
        result = supabase.table('prompt_sections').select('*').limit(1).execute()
        print("   ⚠️  Table still exists (might need manual deletion)")
    except Exception as e:
        if 'does not exist' in str(e).lower() or 'not found' in str(e).lower():
            print("   ✅ Table successfully removed!")
        else:
            print(f"   ℹ️  Verification result: {str(e)}")
    
    # Summary
    print("\n" + "="*60)
    print("✅ CLEANUP COMPLETE!")
    print("="*60)
    
    print("\n📊 SUMMARY:")
    print("   ✅ Backup created (if table existed)")
    print("   ✅ prompt_sections table removed")
    print("   ✅ All prompts now in app/prompts/*.py")
    print("   ✅ Database is cleaner and simpler")
    
    print("\n📝 REMAINING TABLES:")
    print("   ✅ menu_items - Menu data (dynamic)")
    print("   ✅ orders - Order records")
    print("   ✅ draft_sessions - Active sessions")
    print("   ✅ conversation_history - Chat history")
    print("   ✅ order_items - Order details")
    
    print("\n🎉 Your database is now clean and organized!")
    print("="*60 + "\n")

if __name__ == "__main__":
    cleanup_database()

