"""
Check Connection Script
Verifies connection to Evolution API and webhook configuration
"""
import sys
import os
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

import httpx
from app.config import settings

def check_server():
    """Check if local server is running"""
    print("\n" + "="*80)
    print("🔍 CHECKING LOCAL SERVER")
    print("="*80)
    
    try:
        response = httpx.get(f"http://localhost:{settings.SERVER_PORT}/health", timeout=5)
        if response.status_code == 200:
            print(f"✅ Server is running on port {settings.SERVER_PORT}")
            print(f"   Response: {response.json()}")
            return True
        else:
            print(f"⚠️  Server responded with status {response.status_code}")
            return False
    except Exception as e:
        print(f"❌ Cannot connect to local server: {e}")
        print(f"   Make sure server is running on port {settings.SERVER_PORT}")
        return False


def check_evolution_api():
    """Check Evolution API connection"""
    print("\n" + "="*80)
    print("🔍 CHECKING EVOLUTION API")
    print("="*80)
    
    print(f"\n📡 Evolution API URL: {settings.EVOLUTION_API_URL}")
    print(f"🔑 Instance Name: {settings.EVOLUTION_INSTANCE_NAME}")
    
    try:
        # Check instance status
        url = f"{settings.EVOLUTION_API_URL}/instance/connectionState/{settings.EVOLUTION_INSTANCE_NAME}"
        headers = {"apikey": settings.EVOLUTION_API_KEY}
        
        print(f"\n🔗 Checking instance status...")
        response = httpx.get(url, headers=headers, timeout=10)
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Evolution API is reachable")
            print(f"   Instance: {settings.EVOLUTION_INSTANCE_NAME}")
            print(f"   Status: {data}")
            return True
        else:
            print(f"⚠️  Evolution API responded with status {response.status_code}")
            print(f"   Response: {response.text}")
            return False
            
    except Exception as e:
        print(f"❌ Cannot connect to Evolution API: {e}")
        return False


def check_webhook():
    """Check webhook configuration"""
    print("\n" + "="*80)
    print("🔍 CHECKING WEBHOOK CONFIGURATION")
    print("="*80)
    
    try:
        # Get webhook settings
        url = f"{settings.EVOLUTION_API_URL}/webhook/find/{settings.EVOLUTION_INSTANCE_NAME}"
        headers = {"apikey": settings.EVOLUTION_API_KEY}
        
        print(f"\n🔗 Fetching webhook configuration...")
        response = httpx.get(url, headers=headers, timeout=10)
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Webhook configuration found:")
            print(f"   URL: {data.get('webhook', {}).get('url', 'Not set')}")
            print(f"   Events: {data.get('webhook', {}).get('events', [])}")
            
            webhook_url = data.get('webhook', {}).get('url', '')
            if 'webhook/whatsapp' in webhook_url:
                print(f"\n✅ Webhook is configured correctly")
                return True
            else:
                print(f"\n⚠️  Webhook URL might be incorrect")
                print(f"   Expected: http://your-server:8000/webhook/whatsapp")
                print(f"   Current: {webhook_url}")
                return False
        else:
            print(f"⚠️  Could not fetch webhook config (status {response.status_code})")
            print(f"   Response: {response.text}")
            return False
            
    except Exception as e:
        print(f"❌ Error checking webhook: {e}")
        return False


def test_webhook():
    """Test webhook endpoint"""
    print("\n" + "="*80)
    print("🔍 TESTING WEBHOOK ENDPOINT")
    print("="*80)
    
    try:
        # Send test message to webhook
        url = f"http://localhost:{settings.SERVER_PORT}/webhook/whatsapp"
        
        test_payload = {
            "event": "messages.upsert",
            "instance": settings.EVOLUTION_INSTANCE_NAME,
            "data": {
                "key": {
                    "remoteJid": "201556307784@s.whatsapp.net",
                    "fromMe": False,
                    "id": "test-message-id"
                },
                "message": {
                    "conversation": "test message"
                },
                "messageType": "conversation"
            }
        }
        
        print(f"\n📤 Sending test message to webhook...")
        response = httpx.post(url, json=test_payload, timeout=10)
        
        if response.status_code == 200:
            print(f"✅ Webhook endpoint is working")
            print(f"   Response: {response.json()}")
            return True
        else:
            print(f"⚠️  Webhook responded with status {response.status_code}")
            print(f"   Response: {response.text}")
            return False
            
    except Exception as e:
        print(f"❌ Error testing webhook: {e}")
        return False


def main():
    """Run all checks"""
    print("\n" + "="*80)
    print("🚀 CONNECTION DIAGNOSTIC TOOL")
    print("="*80)
    
    results = {
        "server": False,
        "evolution_api": False,
        "webhook_config": False,
        "webhook_test": False
    }
    
    # Run checks
    results["server"] = check_server()
    results["evolution_api"] = check_evolution_api()
    results["webhook_config"] = check_webhook()
    results["webhook_test"] = test_webhook()
    
    # Summary
    print("\n" + "="*80)
    print("📊 DIAGNOSTIC SUMMARY")
    print("="*80)
    
    print(f"\n{'✅' if results['server'] else '❌'} Local Server: {'Running' if results['server'] else 'Not Running'}")
    print(f"{'✅' if results['evolution_api'] else '❌'} Evolution API: {'Connected' if results['evolution_api'] else 'Not Connected'}")
    print(f"{'✅' if results['webhook_config'] else '❌'} Webhook Config: {'OK' if results['webhook_config'] else 'Issue Found'}")
    print(f"{'✅' if results['webhook_test'] else '❌'} Webhook Test: {'Passed' if results['webhook_test'] else 'Failed'}")
    
    all_ok = all(results.values())
    
    if all_ok:
        print("\n🎉 All checks passed! System is ready.")
    else:
        print("\n⚠️  Some checks failed. Please review the issues above.")
        print("\n📋 Troubleshooting steps:")
        
        if not results["server"]:
            print("   1. Start the server: python -m uvicorn app.main:app --host 0.0.0.0 --port 8000")
        
        if not results["evolution_api"]:
            print("   2. Check Evolution API URL and API key in .env file")
            print("   3. Verify Evolution API instance is running")
        
        if not results["webhook_config"]:
            print("   4. Configure webhook in Evolution API")
            print("   5. Set webhook URL to: http://your-server:8000/webhook/whatsapp")
        
        if not results["webhook_test"]:
            print("   6. Check server logs for errors")
            print("   7. Verify webhook endpoint is accessible")
    
    print("\n" + "="*80)


if __name__ == "__main__":
    main()

