"""
Check all database tables schema
"""
import os
import sys
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from supabase import create_client
from dotenv import load_dotenv

load_dotenv()

supabase = create_client(
    os.getenv('SUPABASE_URL'),
    os.getenv('SUPABASE_KEY')
)

def print_section(title):
    """Print section header"""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80)

# List of all tables in the system
TABLES = [
    'orders',
    'order_items',
    'chats',
    'draft_sessions',
    'menu_items',
    'conversation_history',
    'customers',
    'delivery_addresses',
]

print_section("📊 DATABASE SCHEMA AUDIT")

for table_name in TABLES:
    print(f"\n{'='*80}")
    print(f"  📋 TABLE: {table_name}")
    print(f"{'='*80}")
    
    try:
        result = supabase.table(table_name).select('*').limit(1).execute()
        
        if result.data and len(result.data) > 0:
            row = result.data[0]
            print(f"\n✅ Columns in '{table_name}':")
            
            # Sort columns alphabetically
            for key in sorted(row.keys()):
                value = row[key]
                value_type = type(value).__name__
                
                # Show sample value (truncated)
                if value is None:
                    sample = "NULL"
                elif isinstance(value, str):
                    sample = f'"{value[:30]}..."' if len(str(value)) > 30 else f'"{value}"'
                elif isinstance(value, (list, dict)):
                    sample = f"{value_type} (length: {len(value)})"
                else:
                    sample = str(value)
                
                print(f"   • {key:30s} ({value_type:10s}) = {sample}")
        else:
            print(f"\n⚠️  Table '{table_name}' is empty (no data to show schema)")
            
    except Exception as e:
        print(f"\n❌ Error accessing table '{table_name}': {e}")

print_section("📊 SUMMARY")
print(f"\n✅ Schema audit complete for {len(TABLES)} tables")
print("\n💡 Use this information to verify column names in your code")

